import {PureComponent, ReactNode} from 'react';
import _chunk from 'lodash/chunk';

import {IRoomAmenity} from 'types/hotels/offer/IHotelRoom';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {deviceModDesktop, deviceModMobile} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelPageRooms';

import LinkButton from 'components/LinkButton/LinkButton';

import Amenity from '../Amenity/Amenity';

/* Styles */
import cx from './Amenities.scss';

/* Types */
export interface IHotelPageRoomAmenitiesProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    amenities?: IRoomAmenity[];
    amenityLimit?: number;
    viewType: 'table' | 'column' | 'inline';
    amenityTextSize: 's' | 'm';
    amenityFontWeight: 'normal' | 'medium' | 'bold';
    handleMoreInfoClick?: () => void;
    renderMoreButton?: boolean;
    itemsInRow?: 3 | 4;
}

/* Constants */
const TOTAL_ITEMS_IN_ROW = 4;

class Amenities extends PureComponent<IHotelPageRoomAmenitiesProps> {
    private renderPageRoomAmenity = (amenity: IRoomAmenity): ReactNode => {
        const {itemsInRow = TOTAL_ITEMS_IN_ROW, deviceType} = this.props;
        const {amenityTextSize, amenityFontWeight} = this.props;

        return (
            <Amenity
                key={amenity.id}
                amenity={amenity}
                className={cx(
                    'amenity',
                    `amenity${deviceType.isMobile ? '' : itemsInRow}`,
                )}
                amenityTextSize={amenityTextSize}
                amenityFontWeight={amenityFontWeight}
            />
        );
    };

    private getInlineViewAmenitiesNode(): ReactNode {
        const {amenities = [], amenityLimit = 0} = this.props;

        return amenities.slice(0, amenityLimit).map(this.renderPageRoomAmenity);
    }

    private getColumnViewAmenitiesNode(): ReactNode {
        const {amenities = [], amenityLimit} = this.props;

        return amenities.slice(0, amenityLimit).map(this.renderPageRoomAmenity);
    }

    private getTableViewAmenitiesNode(): ReactNode {
        const {amenities = [], itemsInRow = TOTAL_ITEMS_IN_ROW} = this.props;
        const amenityRows = _chunk(amenities, itemsInRow);

        return amenityRows.map((amenityRow, index) => (
            <div key={index} className={cx('amenityRow')}>
                {amenityRow.map(this.renderPageRoomAmenity)}
            </div>
        ));
    }

    private getAmenitiesNodeByType(): ReactNode {
        const {viewType} = this.props;

        switch (viewType) {
            case 'table': {
                return this.getTableViewAmenitiesNode();
            }

            case 'column': {
                return this.getColumnViewAmenitiesNode();
            }

            case 'inline': {
                return this.getInlineViewAmenitiesNode();
            }

            default: {
                return null;
            }
        }
    }

    private renderMoreButton(): ReactNode {
        const {renderMoreButton, handleMoreInfoClick, deviceType} = this.props;

        if (!renderMoreButton) {
            return null;
        }

        if (deviceType.isMobile) {
            return (
                <LinkButton
                    className={cx(
                        'moreButton',
                        deviceModMobile('moreButton', deviceType),
                    )}
                    onClick={handleMoreInfoClick}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'moreButton',
                    })}
                >
                    {i18nBlock.moreInformationAboutRoom()}
                </LinkButton>
            );
        }

        return (
            <LinkButton
                className={cx(
                    'moreButton',
                    deviceModDesktop('moreButton', deviceType),
                )}
                onClick={handleMoreInfoClick}
            >
                {i18nBlock.viewAllAmenities()}
            </LinkButton>
        );
    }

    render(): ReactNode {
        const {className, viewType, amenities} = this.props;

        if (!amenities?.length) {
            return null;
        }

        return (
            <section
                className={cx(
                    'root',
                    {
                        [`root_view_${viewType}`]: Boolean(viewType),
                    },
                    className,
                )}
                {...prepareQaAttributes(this.props)}
            >
                {this.getAmenitiesNodeByType()}
                {this.renderMoreButton()}
            </section>
        );
    }
}

export default Amenities;
