import {FunctionComponent, ReactElement, ReactNode} from 'react';

import {
    ERoomBedIcon,
    IRoomBedConfigurationItem,
} from 'types/hotels/offer/IHotelRoom';
import {IWithClassName} from 'types/withClassName';

import {CHAR_NBSP} from 'utilities/strings/charCodes';

/* Components */
import SingleBedIcon from 'icons/16/SingleBed';
import DoubleBedIcon from 'icons/16/DoubleBed';

/* Init styles */
import cx from './Bed.scss';

/* Types */
export interface IBedProps extends IWithClassName {
    bed: IRoomBedConfigurationItem;
    canViewBedName?: boolean;
    canViewImage?: boolean;
}

/* Constants */
const MAX_ICON_COUNT = 2;

/* Helpers */
const getIconById = (icon?: ERoomBedIcon): ReactElement | null => {
    switch (icon) {
        case ERoomBedIcon.SINGLE_BED: {
            return <SingleBedIcon className={cx('icon')} />;
        }

        case ERoomBedIcon.DOUBLE_BED: {
            return <DoubleBedIcon className={cx('icon')} />;
        }

        default: {
            return null;
        }
    }
};

const getRoomBedIconBlockById = ({
    icon,
    quantity,
}: {
    icon?: ERoomBedIcon;
    quantity: number;
}): ReactNode => {
    const iconNode = getIconById(icon);

    if (!iconNode) {
        return null;
    }

    switch (true) {
        case quantity > MAX_ICON_COUNT: {
            return (
                <>
                    {`${quantity}${CHAR_NBSP}x${CHAR_NBSP}`}
                    {iconNode}
                </>
            );
        }

        case quantity === MAX_ICON_COUNT: {
            return (
                <>
                    {iconNode}
                    {iconNode}
                </>
            );
        }

        default: {
            return iconNode;
        }
    }
};

const Bed: FunctionComponent<IBedProps> = ({
    className,
    bed,
    canViewBedName = true,
    canViewImage = true,
}) => {
    const {icon, quantity, nameInflectedForm} = bed;
    const iconBlockNode =
        canViewImage && getRoomBedIconBlockById({icon, quantity});
    const canUseQuantity = quantity <= MAX_ICON_COUNT;
    const isSingleBedIcon = icon === ERoomBedIcon.SINGLE_BED;

    return (
        <div
            className={cx(
                'root',
                {
                    root_hasSingleBedIcon: isSingleBedIcon,
                },
                className,
            )}
        >
            {Boolean(iconBlockNode) && (
                <div className={cx('iconBlock')}>{iconBlockNode}</div>
            )}
            {canViewBedName && (
                <div className={cx('name')}>
                    {`${
                        canUseQuantity ? `${quantity}${CHAR_NBSP}` : ''
                    }${nameInflectedForm}`}
                </div>
            )}
        </div>
    );
};

export default Bed;
