import {
    ReactNode,
    FunctionComponent,
    useCallback,
    useState,
    useEffect,
    useRef,
} from 'react';

import {IHotelBoyMetaRoomWithOffers} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/types';
import {
    EHotelBadgeType,
    IHotelPartnerOffer,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import scrollToNode from 'utilities/dom/scrollToNode';
import getOffersCollapsedInfo from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/utilities/getOffersCollapsedInfo';
import {parseImageUrlTemplate} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getGuestsCount from 'projects/depreacted/hotels/pages/HotelPage/utilities/getGuestsCount';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/hotels-HotelPageRooms';
import * as i18nOffersBlock from 'i18n/hotels-HotelPageOffersInfo';

/* Components */
import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import Separator from 'components/Separator/Separator';
import TravelImage from 'components/TravelImage/TravelImage';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import HotelImageStub from 'projects/depreacted/hotels/components/HotelImageStub/HotelImageStub';
import HotelOfferBadges from 'projects/depreacted/hotels/components/HotelOfferBadges/HotelOfferBadges';
import Offer from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/Offer/Offer';
import MainOffers from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/MainOffers/MainOffers';
import OfferLabels from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/OfferLabels/OfferLabels';
import HotelPageImagesViewer from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageImagesViewer/HotelPageImagesViewer';
import RoomSize from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/Rooms/components/RoomSize/RoomSize';
import BedGroups from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/Rooms/components/BedGroups/BedGroups';
import Amenities from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/Rooms/components/Amenities/Amenities';
import HotelPageRoomGalleryRedesign from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageRoomGallery/HotelPageRoomGalleryRedesign';
import HotelPageRoomDetailedInfoButton from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageAllAmenities/blocks/HotelPageRoomDetailedInfo/HotelPageRoomDetailedInfoButton';

/* Styles */

import cx from './Room.scss';

const SHORT_AMENITY_LIMIT = 3;

export interface IRoomProps extends IWithQaAttributes {
    room: IHotelBoyMetaRoomWithOffers;
    partnerOffers?: IHotelPartnerOffer[];
    operatorById?: OperatorByIdType;
    isShort: boolean;
    onOfferSelect?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    onCheckPricesClick?: () => void;
    canShowImages?: boolean;
    nightsCount?: number;
    chosenRoomId?: string;
    searchParams?: IRequiredOfferParams;
}

const Room: FunctionComponent<IRoomProps> = ({
    room,
    partnerOffers,
    operatorById,
    /** TODO: 💀 DeadCode - https://st.yandex-team.ru/TRAVELFRONT-6523 */
    isShort,
    canShowImages,
    onOfferSelect,
    onOfferWatchButtonClick,
    onCheckPricesClick,
    nightsCount,
    chosenRoomId,
    searchParams,
    ...rest
}) => {
    const {id, offers, name, images, mainAmenities, area, bedGroups, badges} =
        room;
    const deviceType = useDeviceType();
    const {isDesktop, isMobile} = deviceType;
    const {
        value: isCheckPricesHovered,
        setTrue: setCheckPricesHovered,
        setFalse: unsetCheckPricesHovered,
    } = useBoolean(false);
    const roomRef = useRef<HTMLElement>(null);
    const checkPricesRef = useRef<HTMLElement>(null);

    const handleWatchButtonClick = useCallback(() => {
        onOfferWatchButtonClick?.(id);
    }, [onOfferWatchButtonClick, id]);

    const [modalImagesInitialIndex, setModalImagesInitialIndex] = useState<
        number | undefined
    >(undefined);

    /* Handlers */

    useEffect(() => {
        if (!isShort && chosenRoomId === id) {
            if (roomRef.current) {
                scrollToNode(roomRef.current, {behavior: 'smooth'});
            }
        }
    }, [isShort, chosenRoomId, id, roomRef]);

    const handleCloseModalImages = useCallback(() => {
        setModalImagesInitialIndex(undefined);
    }, [setModalImagesInitialIndex]);

    const handleImageClick = useCallback(
        (imageIndex: number) => {
            setModalImagesInitialIndex(imageIndex);

            reachGoal(EHotelsGoal.HOTEL_PAGE_OFFERS_TAB_IMAGE_CLICK);
        },
        [setModalImagesInitialIndex],
    );

    /* Render */

    const renderModalImagesViewer = (): ReactNode => {
        return (
            images && (
                <HotelPageImagesViewer
                    isVisible={typeof modalImagesInitialIndex === 'number'}
                    totalImageCount={images.length}
                    initialIndex={modalImagesInitialIndex}
                    name={name}
                    images={images}
                    onCloseClick={handleCloseModalImages}
                />
            )
        );
    };

    function renderOffers(): ReactNode {
        const hasOffers = offers.length > 0;

        if (!searchParams) {
            return (
                <Flex justifyContent="flex-end">
                    {isMobile && (
                        <Separator className={cx('checkPricesSeparator')} />
                    )}
                    <Button
                        theme="primary"
                        onClick={onCheckPricesClick}
                        onMouseEnter={setCheckPricesHovered}
                        onMouseLeave={unsetCheckPricesHovered}
                        innerRef={checkPricesRef}
                        className={cx('checkPricesButton')}
                        size={isMobile ? 'l' : 'm'}
                    >
                        {i18nOffersBlock.checkPrices()}
                    </Button>
                    {isDesktop && (
                        <MessageBoxPopup
                            isVisible={isCheckPricesHovered}
                            anchorRef={checkPricesRef}
                        >
                            <div className={cx('tooltip')}>
                                {i18nOffersBlock.specifyDatesToCheckPrices()}
                            </div>
                        </MessageBoxPopup>
                    )}
                </Flex>
            );
        }

        const guestsCount = getGuestsCount(searchParams);

        if (!hasOffers) {
            return (
                <div
                    className={cx('emptyRoomOffer')}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'emptyOffer',
                    })}
                >
                    <div className={cx('emptyRoomOfferText')}>
                        {i18nBlock.emptyRoomOffersNew({guestsCount})}
                    </div>
                </div>
            );
        }

        if (!isShort) {
            return (
                <MainOffers
                    className={cx('offers')}
                    deviceType={deviceType}
                    offers={offers}
                    partnerOffers={partnerOffers}
                    operatorById={operatorById}
                    isFullOffersTab
                    onOfferSelect={onOfferSelect}
                    nightsCount={nightsCount}
                    isRoomGroup
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'mainOffers',
                    })}
                />
            );
        }

        const {price, badges} = getOffersCollapsedInfo(offers);

        const suitableBadges = badges?.filter(
            ({id: badgeId}) => badgeId !== EHotelBadgeType.HOTEL_DIRECT,
        );

        return (
            <Offer
                className={cx('offer')}
                labels={
                    <OfferLabels
                        mealType={room.pansionAggregate}
                        cancellationType={room.cancellationInfoAggregate}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'offerLabels',
                        })}
                    />
                }
                type="compact"
                onOfferSelect={handleWatchButtonClick}
                nightsCount={nightsCount}
                price={price}
                badges={suitableBadges}
                buttonTheme="primary"
                buttonText={i18nBlock.watch()}
                showNightsCount={Boolean(nightsCount)}
                {...prepareQaAttributes({parent: rest, current: 'offer'})}
            />
        );
    }

    function renderRoomGallery(): ReactNode {
        if (images?.length) {
            return (
                <HotelPageRoomGalleryRedesign
                    className={cx('roomGallery')}
                    deviceType={deviceType}
                    images={images}
                    onImageClick={handleImageClick}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'gallery',
                    })}
                />
            );
        }

        if (isDesktop) {
            return (
                <div className={cx('withoutImageGallery')}>
                    <HotelImageStub
                        className={cx('withoutImage')}
                        captionText={i18nBlock.noImages()}
                    />
                </div>
            );
        }

        return null;
    }

    function renderRoomInfo(): ReactNode {
        return (
            <div className={cx('info')}>
                <Heading
                    level={2}
                    {...prepareQaAttributes({parent: rest, current: 'name'})}
                >
                    {name}
                </Heading>
                <HotelOfferBadges className={cx('badges')} badges={badges} />
                {renderBedGroupsAndSize()}
                {renderMainAmenities()}
                {renderDetailedInfo()}
                {renderOffers()}
            </div>
        );
    }

    function renderMainAmenities(): ReactNode {
        if (mainAmenities?.length) {
            return (
                <Amenities
                    className={cx('mainAmenities')}
                    amenities={mainAmenities}
                    viewType={isDesktop && !isShort ? 'table' : 'inline'}
                    amenityTextSize="m"
                    amenityFontWeight="normal"
                    amenityLimit={
                        isMobile || isShort ? SHORT_AMENITY_LIMIT : undefined
                    }
                    handleMoreInfoClick={handleWatchButtonClick}
                    renderMoreButton={(isDesktop && isShort) || isMobile}
                    deviceType={deviceType}
                    itemsInRow={3}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'amenities',
                    })}
                />
            );
        }

        return null;
    }

    function renderDetailedInfo(): ReactNode {
        if (isMobile) {
            return null;
        }

        return (
            <HotelPageRoomDetailedInfoButton
                className={cx('detailedInfo')}
                room={room}
                handleMoreInfoClick={handleWatchButtonClick}
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'detailedInfoButton',
                })}
            />
        );
    }

    function renderBedGroupsAndSizeContent(): ReactNode {
        const roomSizeNode = area ? (
            <RoomSize className={cx('title')} area={area} />
        ) : null;

        if (roomSizeNode) {
            if (bedGroups?.length) {
                return (
                    <BedGroups
                        className={cx('bedGroups')}
                        bedClassName={cx('bed')}
                        bedGroupClassName={cx('bedGroup')}
                        bedGroups={bedGroups}
                        canViewBedName
                        canViewImage={false}
                        isSeparatorBlock={!isShort}
                    >
                        <DotSeparator />
                        {roomSizeNode}
                    </BedGroups>
                );
            }

            return roomSizeNode;
        }

        if (bedGroups?.length) {
            return (
                <BedGroups
                    className={cx('bedGroups')}
                    bedClassName={cx('bed')}
                    bedGroupClassName={cx('bedGroup')}
                    bedGroups={bedGroups}
                    canViewBedName
                    canViewImage={false}
                    isSeparatorBlock={!isShort}
                />
            );
        }

        return null;
    }

    function renderBedGroupsAndSize(): ReactNode {
        const bedGroupsAndSizeContent = renderBedGroupsAndSizeContent();

        if (bedGroupsAndSizeContent === null) {
            return null;
        }

        return (
            <div
                className={cx('bedGroupsAndSize')}
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'bedGroupsAndSize',
                })}
            >
                {bedGroupsAndSizeContent}
            </div>
        );
    }

    function renderImage(): ReactNode {
        if (images?.length) {
            const image = parseImageUrlTemplate(
                images[0].urlTemplate,
                EHotelImageAltayStandardSize.S,
                images[0].sizes,
            );

            return (
                <div
                    className={cx('imageBlock')}
                    onClick={handleWatchButtonClick}
                >
                    <TravelImage
                        className={cx('image')}
                        src={image}
                        imageClassName={cx('pic')}
                    />
                    <div className={cx('imagesCount')}>
                        <Text
                            size="m"
                            color="current"
                            className={cx('imagesCount__text')}
                        >
                            {i18nBlock.imagesCount({
                                imagesCount: images.length,
                            })}
                        </Text>
                    </div>
                </div>
            );
        }
    }

    function renderTop(): ReactNode {
        return (
            <div className={cx('top')}>
                <div className="name">
                    <Heading
                        className={cx('heading')}
                        level={3}
                        onClick={handleWatchButtonClick}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'topName',
                        })}
                    >
                        {name}
                    </Heading>
                    {renderBedGroupsAndSize()}
                </div>
                {canShowImages && renderImage()}
            </div>
        );
    }

    function renderBottom(): ReactNode {
        return (
            <div className={cx('bottom')}>
                {renderMainAmenities()}
                {renderOffers()}
            </div>
        );
    }

    if (isShort) {
        return (
            <div
                key={id}
                className={cx(
                    'room',
                    'room_short',
                    deviceMods('room', deviceType),
                )}
                {...prepareQaAttributes(rest)}
            >
                {renderTop()}
                {renderBottom()}
            </div>
        );
    }

    return (
        <Card
            key={id}
            className={cx('room', 'room_full', deviceMods('room', deviceType))}
            shadow={isMobile ? 'default' : undefined}
            ref={roomRef}
            {...prepareQaAttributes(rest)}
        >
            {canShowImages && renderRoomGallery()}
            {renderRoomInfo()}
            {renderModalImagesViewer()}
        </Card>
    );
};

export default Room;
