import {PureComponent, ReactNode} from 'react';

import {
    INITIAL,
    IN_PROGRESS,
    AWAITS_PAYMENT,
    NEED_AUTHORIZATION,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';
import {URLs} from 'constants/urls';
import {EProjectName} from 'constants/common';

import getFinishPaymentPagePath from 'projects/depreacted/hotels/utilities/getFinishPaymentPagePath/getFinishPaymentPagePath';
import {internalUrl} from 'utilities/url';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-StartDeferredPaymentPage';

import HotelsBookLayout from 'projects/depreacted/hotels/components/HotelsBookLayout/HotelsBookLayout';
import BookLoader from 'components/BookLoader/BookLoader';
import CheckOrderAuthorization from 'components/CheckOrderAuthorization/CheckOrderAuthorizationContainer';

import {THotelsStartDeferredPaymentPageContainerProps} from 'projects/depreacted/hotels/pages/HotelsStartDeferredPaymentPage/HotelsStartDeferredPaymentPageContainer';

import cx from './HotelsStartDeferredPaymentPage.scss';

interface IHotelsStartDeferredPaymentProps
    extends THotelsStartDeferredPaymentPageContainerProps,
        IWithQaAttributes {}

class HotelsStartDeferredPayment extends PureComponent<IHotelsStartDeferredPaymentProps> {
    static defaultProps = {
        ...prepareQaAttributes('startDeferredPaymentPage'),
    };

    componentDidUpdate(prevProps: IHotelsStartDeferredPaymentProps): void {
        this.checkStartPaymentIfNeed(prevProps);
        this.checkLeavePageIfNeed();
    }

    /* Handlers */

    handleSuccessOrderAuthorization = (): void => {
        const {startOrderPolling, location} = this.props;
        const {orderId} = getQueryByLocation(location);

        if (orderId) {
            startOrderPolling({orderId: String(orderId)});
        }
    };

    /* Helpers */

    checkStartPaymentIfNeed(prevProps: IHotelsStartDeferredPaymentProps): void {
        const {orderInfo, startPayment} = this.props;
        const prevMayBePaymentStarted =
            prevProps?.orderInfo?.payment?.mayBeStarted;
        const mayBePaymentStarted = orderInfo?.payment?.mayBeStarted;

        if (
            orderInfo?.orderId &&
            mayBePaymentStarted &&
            !prevMayBePaymentStarted
        ) {
            startPayment({
                orderId: orderInfo?.orderId,
                finishPaymentPagePath: getFinishPaymentPagePath(),
            });
        }
    }

    getRedirectUrl(): string | undefined {
        const {orderInfo} = this.props;
        const orderId = orderInfo?.orderId;
        const orderStatus = orderInfo?.status;
        const currentPayment = orderInfo?.payment?.current;
        const mayBePaymentStarted = orderInfo?.payment?.mayBeStarted;

        if (
            mayBePaymentStarted ||
            orderStatus === INITIAL ||
            orderStatus === IN_PROGRESS ||
            orderStatus === NEED_AUTHORIZATION
        ) {
            return;
        }

        if (orderStatus === AWAITS_PAYMENT && currentPayment && orderId) {
            return internalUrl(URLs.hotelsBookPayment, {orderId});
        }

        if (orderId) {
            // return getOrderUrl(orderId);
        }

        return URLs[EProjectName.NOT_FOUND];
    }

    checkLeavePageIfNeed(): void {
        const {history} = this.props;
        const redirectUrl = this.getRedirectUrl();

        if (redirectUrl) {
            history.push(redirectUrl);
        }
    }

    /* Render */

    renderContent(): ReactNode {
        const {orderInfo, location} = this.props;
        const orderStatus = orderInfo?.status;
        const {orderId} = getQueryByLocation(location);

        if (orderId && orderStatus === NEED_AUTHORIZATION) {
            return (
                <CheckOrderAuthorization
                    className={cx('checkOrderAuthorization')}
                    orderId={String(orderId)}
                    onSuccessOrderAuthorization={
                        this.handleSuccessOrderAuthorization
                    }
                />
            );
        }

        return (
            <BookLoader
                isLoading
                title={i18nBlock.loaderTitle()}
                description={i18nBlock.loaderDescription()}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'pageLoader',
                })}
            />
        );
    }

    render(): ReactNode {
        const {deviceType} = this.props;

        return (
            <HotelsBookLayout
                deviceType={deviceType}
                hasLoader
                {...prepareQaAttributes(this.props)}
            >
                {this.renderContent()}
            </HotelsBookLayout>
        );
    }
}

export default HotelsStartDeferredPayment;
