import React from 'react';
import _noop from 'lodash/noop';
import moment from 'moment';

import {EProjectName} from 'constants/common';
import {
    SOLOMON_HOTELS_PAYMENT_PAGE_LOADED,
    SOLOMON_HOTELS_TRUST_ERROR,
    SOLOMON_HOTELS_TRUST_LOADED,
    SOLOMON_HOTELS_TRUST_SUCCESS,
} from 'constants/solomon';
import {
    AWAITS_PAYMENT,
    PAYMENT_FAILED,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import EOriginalPaymentErrorCode from 'types/common/EOriginalPaymentErrorCode';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {prepareDeferredPaymentDateTime} from 'projects/depreacted/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';
import {
    HUMAN,
    HUMAN_DATETIME_WITHOUT_YEAR,
    ROBOT,
} from 'utilities/dateUtils/formats';
import {reachGoal} from 'utilities/metrika';
import {sendPaymentCounter} from 'utilities/solomon';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {logError} from 'utilities/logger/logError';
import getFinishPaymentPagePath from 'projects/depreacted/hotels/utilities/getFinishPaymentPagePath/getFinishPaymentPagePath';
import {CHAR_EN_DASH} from 'utilities/strings/charCodes';
import {getHotelBySlugUrl} from 'projects/depreacted/hotels/utilities/urls/getHotelBySlugUrl';

import * as i18nBlock from 'i18n/hotels-BookPaymentPage';
import * as i18nPluralBlock from 'i18n/common-plural';

import BookingMailSubscription from 'containers/BookingMailSubscription/BookingMailSubscription';

import PaymentLayout from 'components/Layouts/NewPaymentLayout/PaymentLayout';
import PaymentIframe from 'components/PaymentIframe/PaymentIframe';
import Avatar from 'components/Layouts/PaymentLayout/components/PaymentHeader/components/Avatar/Avatar';
import Price from 'components/Price/Price';
import BookLoader from 'components/BookLoader/BookLoader';
import BookPaymentFailedModal from 'components/BookPaymentFailedModal/BookPaymentFailedModal';

import BookRedirectController from 'projects/depreacted/hotels/containers/BookRedirectController/BookRedirectController';
import {TPaymentPageProps} from 'projects/depreacted/hotels/pages/PaymentPage/PaymentPageContainer';

import cx from './PaymentPage.scss';

const PAYMENT_PAGE_QA = 'hotels-payment-page';

interface IPaymentPageState {
    loaded: boolean;
    paid: boolean;
    paymentIsStarted: boolean;
    canRenderFailedModal: boolean;
}

class PaymentPage extends React.Component<
    TPaymentPageProps,
    IPaymentPageState
> {
    static defaultProps = {
        startOrderPolling: _noop,
    };

    state: IPaymentPageState = {
        loaded: false,
        paid: false,
        paymentIsStarted: false,
        canRenderFailedModal: true,
    };

    componentDidMount(): void {
        this.startOrderPolling();

        reachGoal(EHotelsGoal.PAYMENT_PAGE_LOAD);
        sendPaymentCounter(SOLOMON_HOTELS_PAYMENT_PAGE_LOADED);
    }

    /* Polling actions */

    protected startOrderPolling(): void {
        const {location, startOrderPolling} = this.props;
        const {orderId} = getQueryByLocation(location);

        if (orderId && typeof orderId === 'string') {
            startOrderPolling({orderId});
        }
    }

    /* Handlers and actions */

    protected handleFrameLoaded = (): void => {
        this.setState({loaded: true});

        reachGoal(EHotelsGoal.PAYMENT_PAGE_SHOWN);
    };

    protected handlePaymentFormLoaded = (): void => {
        /**
         * Останавливаем поллинг до момента пока пользователь завершит оплату
         */
        this.props.stopOrderPolling();
    };

    protected handleStartPayment = (): void => {
        reachGoal(EHotelsGoal.PAYMENT_START);

        this.setState({
            canRenderFailedModal: true,
            paymentIsStarted: true,
        });
    };

    protected handleFinishPayment = (): void => {
        this.startOrderPolling();

        this.setState({paid: true});
    };

    protected handlePaymentSuccess = (): void => {
        reachGoal(EHotelsGoal.PAYMENT_SUCCESS);
    };

    protected handlePaymentError = (paymentError: string): void => {
        reachGoal(EHotelsGoal.PAYMENT_ERROR, {hotels: {paymentError}});
    };

    protected handleCancelPaymentClick = (): void => {
        const {
            orderInfo: {orderId, payment},
            offerInfoByToken: {offerInfo},
            removeOrder,
        } = this.props;

        if (orderId && offerInfo) {
            const mayBeCancelled = payment?.mayBeCancelled;

            if (mayBeCancelled) {
                removeOrder({
                    orderId,
                    token: offerInfo.offerOrderInfo.token,
                    label: offerInfo.offerOrderInfo.label,
                });
            } else {
                this.goToOrderPage(orderId);
            }

            this.setState({canRenderFailedModal: false});
        } else {
            logError({
                message:
                    '[YATRAVEL][HOTELS] Ошибка отмены заказа при ошибке оплаты из-за отсутствия данных о предложении или заказе',
                block: 'PaymentPage',
            });
        }
    };

    protected handleRetryPaymentClick = (): void => {
        const {
            orderInfo: {orderId},
            startPayment,
        } = this.props;

        if (orderId) {
            startPayment({
                orderId,
                finishPaymentPagePath: getFinishPaymentPagePath(),
            });
            this.setState({
                loaded: false,
                paid: false,
                canRenderFailedModal: false,
            });
        }
    };

    protected handleRetryUrlClick = (): void => {
        this.setState({
            loaded: false,
            paid: false,
            canRenderFailedModal: false,
        });
    };

    /* Helpers and utilities */

    protected handle3dsPaymentStart(): void {
        reachGoal(EHotelsGoal.PAYMENT_3DS);
    }

    protected prepareDate(date: string): string {
        return moment(date, ROBOT).format(HUMAN);
    }

    protected getOrderInfoDescription(
        hotelInfo: any,
        searchParams: any,
    ): {general: React.ReactNode; additional: React.ReactNode} {
        const {startDate, endDate, adults, childrenAges = []} = searchParams;
        const guestsCount = adults + childrenAges.length;

        return {
            general: (
                <span
                    {...prepareQaAttributes({
                        parent: PAYMENT_PAGE_QA,
                        current: 'hotelName',
                    })}
                >
                    {hotelInfo.name}
                </span>
            ),
            additional: (
                <span
                    {...prepareQaAttributes({
                        parent: PAYMENT_PAGE_QA,
                        current: 'datesAndGuests',
                    })}
                >
                    {this.prepareDate(startDate)} {CHAR_EN_DASH}{' '}
                    {this.prepareDate(endDate)}{' '}
                    {i18nPluralBlock.guests({count: guestsCount})}
                </span>
            ),
        };
    }

    protected checkPaymentOrderStatus(): boolean {
        const {
            orderInfo: {status},
        } = this.props;

        return status === AWAITS_PAYMENT;
    }

    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    protected goToOrderPage(orderId: string): void {
        // const {history} = this.props;
        // history.push(accountURLs.getOrderUrl(orderId));
    }

    protected canRenderPaymentFailModal(): boolean {
        const {
            orderInfo: {status},
        } = this.props;
        const {canRenderFailedModal} = this.state;

        return canRenderFailedModal && status === PAYMENT_FAILED;
    }

    protected getOrderId(): string | undefined {
        const {
            orderInfo: {confirmationInfo},
        } = this.props;

        if (!confirmationInfo) {
            return;
        }

        return confirmationInfo.yandexOrderId;
    }

    protected getIsLoading(): boolean {
        const {
            offerInfoByToken: {isLoading},
        } = this.props;
        const {loaded, paid} = this.state;

        const isPaymentOrderStatus = this.checkPaymentOrderStatus();

        return !loaded || paid || isLoading || !isPaymentOrderStatus;
    }

    protected renderPaymentLoader(): React.ReactNode {
        return (
            <BookLoader
                title={i18nBlock.paymentDotTitle()}
                description={i18nBlock.paymentDotDescription()}
                isModalView={true}
                isLoading={true}
                {...prepareQaAttributes({
                    parent: PAYMENT_PAGE_QA,
                    current: 'loader',
                })}
            />
        );
    }

    getRetryUrl(): string | undefined {
        const {
            offerInfoByToken: {offerInfo},
        } = this.props;

        if (!offerInfo) {
            return undefined;
        }

        const {hotelInfo, searchParams} = offerInfo;
        const [regionSlug, hotelSlug] = hotelInfo.hotelSlug.split('/');
        const {
            startDate: checkinDate,
            endDate: checkoutDate,
            adults,
            childrenAges,
        } = searchParams;

        return getHotelBySlugUrl(
            {regionSlug, hotelSlug},
            {
                checkinDate,
                checkoutDate,
                adults,
                childrenAges: childrenAges?.join(','),
            },
            'HOTEL_OFFERS',
        );
    }

    private isPaymentTimeout(): boolean {
        const {
            orderInfo: {payment},
        } = this.props;
        const {errorCode} = payment || {};

        return errorCode === EOriginalPaymentErrorCode.PAYMENT_TIMEOUT;
    }

    protected renderPaymentFailedModal(): React.ReactNode {
        const {
            orderInfo: {payment},
        } = this.props;
        const {errorCode} = payment || {};
        const isPaymentTimeout = this.isPaymentTimeout();
        let retryUrl;
        let retryClickHandler;

        if (isPaymentTimeout) {
            retryUrl = this.getRetryUrl();
            retryClickHandler = this.handleRetryUrlClick;
        } else {
            retryClickHandler = this.handleRetryPaymentClick;
        }

        return (
            <BookPaymentFailedModal
                onRetryClick={retryClickHandler}
                onCancelClick={this.handleCancelPaymentClick}
                errorCode={errorCode}
                visible={this.canRenderPaymentFailModal()}
                price={isPaymentTimeout ? undefined : payment?.error?.amount}
                retryUrl={retryUrl}
                {...prepareQaAttributes({
                    parent: PAYMENT_PAGE_QA,
                    current: 'errorModal',
                })}
            />
        );
    }

    private renderOfferInfo(): React.ReactNode {
        const {
            offerInfoByToken: {offerInfo},
            orderInfo: {payment},
        } = this.props;

        if (!offerInfo || !payment?.current) {
            return null;
        }

        const {hotelInfo, searchParams} = offerInfo;

        return (
            <PaymentLayout.PaymentOrderInfo
                description={this.getOrderInfoDescription(
                    hotelInfo,
                    searchParams,
                )}
                orderId={this.getOrderId()}
                orderType={EProjectName.HOTELS}
                advantages={this.renderAdvantages()}
            />
        );
    }

    private renderDeferredPaymentLabelIfCan(): React.ReactNode {
        const {orderInfo} = this.props;
        const nextPayment = orderInfo?.payment?.next;

        if (!nextPayment) {
            return;
        }

        return (
            <span
                {...prepareQaAttributes({
                    parent: PAYMENT_PAGE_QA,
                    current: 'deferredPaymentLabel',
                })}
            >
                {insertJSXIntoKey(i18nBlock.deferredPaymentOnYandex)({
                    priceNode: (
                        <Price
                            value={nextPayment.amount.value}
                            currency={nextPayment.amount.currency}
                            {...prepareQaAttributes({
                                parent: PAYMENT_PAGE_QA,
                                current: 'nextPaymentPrice',
                            })}
                        />
                    ),
                    dateNode: (
                        <span
                            className={cx('dateLabel')}
                            {...prepareQaAttributes({
                                parent: PAYMENT_PAGE_QA,
                                current: 'paymentEndsAt',
                            })}
                        >
                            {prepareDeferredPaymentDateTime(
                                nextPayment.paymentEndsAt,
                                HUMAN_DATETIME_WITHOUT_YEAR,
                            )}
                        </span>
                    ),
                })}
            </span>
        );
    }

    private renderAdvantages(): React.ReactNode[] {
        const advantages: React.ReactNode[] = [];
        const deferredPaymentLabel = this.renderDeferredPaymentLabelIfCan();

        if (deferredPaymentLabel) {
            advantages.push(deferredPaymentLabel);
        }

        return advantages;
    }

    render(): React.ReactNode {
        const {
            deviceType,
            orderInfo: {payment, guestsInfo},
        } = this.props;
        const {paymentIsStarted} = this.state;
        const paymentUrl = payment?.current?.paymentUrl;
        const {isDesktop} = deviceType;

        return (
            <>
                <BookRedirectController />
                <PaymentLayout
                    header={
                        isDesktop && (
                            <PaymentLayout.PaymentHeader
                                rightSide={<Avatar />}
                            />
                        )
                    }
                    footer={
                        guestsInfo ? (
                            <BookingMailSubscription
                                email={guestsInfo.customerEmail}
                                source={'ESubscriptionSource.PAYMENT'}
                                subscriptionIsAvailable={paymentIsStarted}
                                travelVerticalName={
                                    'ESubscriptionVerticalName.Hotels'
                                }
                                promoSubscriptionCode={
                                    'ESubscriptionCode.travelNews'
                                }
                            />
                        ) : null
                    }
                    orderInfo={this.renderOfferInfo()}
                    loading={this.getIsLoading()}
                    spinnerText={i18nBlock.orderLoadingDotTitle()}
                    {...prepareQaAttributes(PAYMENT_PAGE_QA)}
                >
                    {paymentUrl && (
                        <PaymentIframe
                            isNewTrust
                            loadEvent={SOLOMON_HOTELS_TRUST_LOADED}
                            successEvent={SOLOMON_HOTELS_TRUST_SUCCESS}
                            errorEvent={SOLOMON_HOTELS_TRUST_ERROR}
                            paymentUrl={paymentUrl}
                            onLoad={this.handleFrameLoaded}
                            onFrameLoaded={this.handlePaymentFormLoaded}
                            onStartPayment={this.handleStartPayment}
                            onStart3dsPayment={this.handle3dsPaymentStart}
                            onFinishPayment={this.handleFinishPayment}
                            onPaymentSuccess={this.handlePaymentSuccess}
                            onPaymentError={this.handlePaymentError}
                            {...prepareQaAttributes({
                                parent: PAYMENT_PAGE_QA,
                                current: 'iframe',
                            })}
                        />
                    )}
                </PaymentLayout>
                {this.renderPaymentFailedModal()}
            </>
        );
    }
}

export default PaymentPage;
