import {ReactNode} from 'react';
import _throttle from 'lodash/throttle';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {SearchPageLayoutView} from 'types/hotels/search/ISearchLayout';

/* Utilities */
import {reachGoal} from 'utilities/metrika';
import scrollTo from 'utilities/dom/scrollTo';

import * as i18nBlock from 'i18n/hotels-SearchPage';

/* Components */
import SearchPageMap from './components/SearchPageMap/SearchPageMap';
import Text from 'components/Text/Text';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import ForceLogin from 'projects/depreacted/hotels/components/ForceLogin/ForceLogin';

import BaseSearchPage from '../BaseSearchPage/BaseSearchPage';
import DesktopSearchPageLayout from '../DesktopSearchPageLayout/DesktopSearchPageLayout';

/* Init styles */
import cx from './DesktopSearchPage.scss';

/* Constants */
const TOGGLE_BUTTON_CLICK_THROTTLE_TIME = 1000;

class DesktopSearchPage extends BaseSearchPage {
    /* Handlers */
    private handleClickToggleViewButton = _throttle(() => {
        scrollTo({top: 0});
        this.toggleActiveView();

        reachGoal(EHotelsGoal.SEARCH_PAGE_MAP_VIEW_CHANGE);
    }, TOGGLE_BUTTON_CLICK_THROTTLE_TIME);

    /* Render */
    private renderMap(): ReactNode {
        const {activeView} = this.state;
        const {windowHeight} = this.props;
        const hasFullScreenMap = activeView === SearchPageLayoutView.MAP;

        return (
            <SearchPageMap
                windowHeight={windowHeight}
                hasFullScreenMap={hasFullScreenMap}
                onToggleMapView={this.handleClickToggleViewButton}
                renderHotelsMap={this.renderHotelsMap}
            />
        );
    }

    private renderSortAndTextFilter(): ReactNode {
        const {hasSort} = this.props;

        return (
            <div className={cx('sortAndTextFilter')}>
                {hasSort ? (
                    <Text size="s" className={cx('sortFirst')}>
                        {i18nBlock.sortFirst()}
                    </Text>
                ) : (
                    <TextSkeleton
                        size="s"
                        className={cx('sortFirst', 'sortFirst_skeleton')}
                    />
                )}
                {this.renderSortInfo(cx('sortBar'), cx('sortGroup'))}
            </div>
        );
    }

    render(): ReactNode {
        const {hasBoyOffers, isLoading, hasMirBanner} = this.props;
        const {activeView} = this.state;
        const hasHotelList = activeView === SearchPageLayoutView.LIST;

        return (
            <DesktopSearchPageLayout
                className={cx('desktopSearchPage')}
                hideSearchInformation
            >
                {this.renderPreloadChunksControl()}
                {this.renderHotelsSearchController()}
                {this.renderSearchInformationProvider()}
                <DesktopSearchPageLayout.Content>
                    {hasHotelList && (
                        <DesktopSearchPageLayout.LeftColumn>
                            {this.renderSearchForm()}
                            {this.renderFilters()}
                            <div className={cx('hotelListAndMap')}>
                                <div
                                    className={cx('hotelListColumn', {
                                        hotelListColumn_onlySearchInformation:
                                            !hasHotelList,
                                    })}
                                >
                                    {hasMirBanner
                                        ? this.renderMirCashbackBanner(
                                              cx('banner'),
                                          )
                                        : this.renderAdFoxBanner(cx('banner'))}
                                    {this.renderSortAndTextFilter()}
                                    <ForceLogin
                                        className={cx('forceLogin')}
                                        isLoading={isLoading}
                                        isSearchPage
                                        hasBoyOffers={hasBoyOffers}
                                    />
                                    {this.renderHotelsList(cx('hotelList'))}
                                </div>
                            </div>
                            {this.renderSearchNavigationButtons()}
                            <div className={cx('disclaimer')}>
                                {this.renderPromoDisclaimer()}
                            </div>
                        </DesktopSearchPageLayout.LeftColumn>
                    )}

                    <DesktopSearchPageLayout.RightColumn
                        noScroll={!hasHotelList}
                        withLargeMap={!hasHotelList}
                        stickToTop
                    >
                        {this.renderMap()}
                    </DesktopSearchPageLayout.RightColumn>
                </DesktopSearchPageLayout.Content>
            </DesktopSearchPageLayout>
        );
    }
}

export default DesktopSearchPage;
