import React, {Component, ReactElement, ReactNode} from 'react';
import {StickyContainer, Sticky, StickyChildArgs} from 'react-sticky';
import {isNumber as _isNumber} from 'lodash';

import {FIXED_FILTERS_BAR_HEIGHT, FIXED_HEADER_HEIGHT} from 'constants/sizes';

import {IWithClassName} from 'types/withClassName';

import {checkNativeStickySupport} from 'utilities/checkNativeStickySupport';

import cx from './StickyColumn.scss';

/* Component Types */
export interface IStickyColumnProps extends IWithClassName {
    noScroll: boolean;
    stickToTop?: boolean;
}

export interface IStickyColumnState {
    isNativeSticky: boolean;
}

class StickyColumn extends Component<IStickyColumnProps, IStickyColumnState> {
    state: IStickyColumnState = {
        isNativeSticky: true,
    };

    componentDidMount(): void {
        const isNativeSticky = checkNativeStickySupport();

        this.setState({isNativeSticky});
    }

    /* Helpers */

    private getColumnHeightStyle = (): string => {
        return `calc(100vh - ${this.getStickyContentOffset()}px)`;
    };

    private getStickyContentOffset(): number {
        const {stickToTop} = this.props;
        const filtersHeaderHeight = stickToTop ? 0 : FIXED_FILTERS_BAR_HEIGHT;

        return FIXED_HEADER_HEIGHT + filtersHeaderHeight;
    }

    /* Render */

    renderReactStickyColumn(mapStyle?: React.CSSProperties): ReactNode {
        const {noScroll} = this.props;

        return (
            <div className={cx('reactStickyColumn')} style={mapStyle}>
                <Sticky
                    disableCompensation={noScroll}
                    topOffset={noScroll ? 0 : -this.getStickyContentOffset()}
                >
                    {this.renderReactSticky}
                </Sticky>
            </div>
        );
    }

    private renderReactSticky = (
        stickyProps: StickyChildArgs,
    ): ReactElement => {
        const {noScroll, children} = this.props;
        const height = this.getColumnHeightStyle();
        const {style = {}, distanceFromBottom = 0} = stickyProps || {};
        const {top} = style;

        const stickyColumnStyle: React.CSSProperties = {
            ...style,
            height,
        };

        if (_isNumber(top)) {
            stickyColumnStyle.top = Math.min(
                this.getStickyContentOffset(),
                distanceFromBottom,
            );
        }

        return (
            <div
                className={cx('reactSticky')}
                style={noScroll ? {height} : stickyColumnStyle}
            >
                {children}
            </div>
        );
    };

    render(): ReactNode {
        const {isNativeSticky} = this.state;
        const {className, children} = this.props;
        const height = this.getColumnHeightStyle();

        if (isNativeSticky) {
            return (
                <div
                    className={cx('nativeSticky', className)}
                    style={{
                        height,
                        top: this.getStickyContentOffset(),
                    }}
                >
                    {children}
                </div>
            );
        }

        return (
            <StickyContainer className={cx('reactStickyContainer', className)}>
                {this.renderReactStickyColumn()}
            </StickyContainer>
        );
    }
}

export default StickyColumn;
