import React, {ReactNode, PureComponent, createRef} from 'react';
import {connect} from 'react-redux';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {
    IChangeFilterGroupPayload,
    IChangePriceFilterPayload,
    IChangeFilterGroupPayloadWithTarget,
    IChangePriceFilterPayloadWithTarget,
    IQuickControl,
    IChangeFilterGroupMeta,
} from 'types/hotels/search/IFiltersInfo';
import {IWithClassName} from 'types/withClassName';

import {hotelsFiltersSelector} from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/selectors/hotelsFiltersSelector';

import {reachGoal} from 'utilities/metrika';
import {deviceMods, deviceModMobile} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {closeFilters} from 'i18n/hotels-SearchPageFilters';

import CloseIcon from 'icons/16/Close';
import Separator from 'components/Separator/Separator';
import HideBodyVerticalScroll from 'components/HideBodyVerticalScroll/HideBodyVerticalScroll';
import HotelsPriceFilter from './components/HotelsPriceFilter/HotelsPriceFilter';
import HotelsFiltersGroups from './components/HotelsFiltersGroups/HotelsFiltersGroups';
import DesktopHotelsFiltersContainer from './components/DesktopHotelsFiltersContainer/DesktopHotelsFiltersContainer';
import MobileHotelsFiltersContainer from './components/MobileHotelsFiltersContainer/MobileHotelsFiltersContainer';
import HotelsFiltersBar from './components/HotelsFiltersBar/HotelsFiltersBar';
import HotelsFiltersControlButtons from './components/HotelsFiltersControlButtons/HotelsFiltersControlButtons';
import HotelsQuickFilters from './components/HotelsQuickFilters/HotelsQuickFilters';
import HotelsFiltersBarSkeleton from './components/HotelsFiltersBarSkeleton/HotelsFiltersBarSkeleton';
import MobileHotelsFiltersBarSkeleton from './components/MobileHotelsFiltersBarSkeleton/MobileHotelsFiltersBarSkeleton';
import ToggleFiltersVisibilityButton from './components/ToggleFiltersVisibilityButton/ToggleFiltersVisibilityButton';
import MobileHotelsFiltersBarRedesignSkeleton from './components/MobileHotelsFiltersBarRedesignSkeleton/MobileHotelsFiltersBarRedesignSkeleton';
import HotelsPriceFilterWithInputs from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/components/HotelsPriceFilter/HotelsPriceFilterWithInputs';

import cx from './HotelsFilters.scss';

interface IHotelsFiltersProps extends IWithClassName, IWithQaAttributes {
    mapListRadioButtonNode?: ReactNode;
    renderSort?: () => ReactNode;

    onChangeFilterGroup(payload: IChangeFilterGroupPayloadWithTarget): void;
    onChangePriceFilter(payload: IChangePriceFilterPayloadWithTarget): void;
    onResetFilters(): void;
    onRevertFilters(): void;
    onApplyFilters(): void;
}

interface IHotelsFiltersState {
    lockScroll: boolean;
    canRenderFiltersGroups: boolean;
    filtersContainerOffset: number;
}

type THotelsFiltersProps = IHotelsFiltersProps &
    ReturnType<typeof hotelsFiltersSelector>;

class HotelsFilters extends PureComponent<
    THotelsFiltersProps,
    IHotelsFiltersState
> {
    private canApplyFiltersAfterFinishAnimate: boolean = false;

    state: IHotelsFiltersState = {
        lockScroll: false,
        canRenderFiltersGroups: false,
        filtersContainerOffset: 70,
    };

    private rootRef = createRef<HTMLDivElement>();

    /* Handlers -> FiltersGroups */

    private handleApplyFiltersGroups = (): void => {
        const {
            deviceInfo: {isMobile},
        } = this.props;

        this.hideFiltersGroups();

        if (isMobile) {
            this.applyFilters();
        } else {
            this.canApplyFiltersAfterFinishAnimate = true;
        }
    };

    private handleFinishAnimateDesktopFiltersGroups = (): void => {
        if (this.canApplyFiltersAfterFinishAnimate) {
            this.applyFilters();

            this.canApplyFiltersAfterFinishAnimate = false;
        } else {
            this.revertFilters();
        }

        this.setState({
            lockScroll: false,
        });
    };

    private handleBaseCloseFilters = (): void => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_CROSS_CLICK);
    };

    private handleCloseMobileFilters = (): void => {
        this.hideFiltersGroups();
        this.revertFilters();
        this.handleBaseCloseFilters();
    };

    private handleToggleViewFilters = (): void => {
        this.setState(({canRenderFiltersGroups, lockScroll}) => ({
            canRenderFiltersGroups: !canRenderFiltersGroups,
            lockScroll: canRenderFiltersGroups ? lockScroll : !lockScroll,
        }));

        reachGoal(EHotelsGoal.SEARCH_PAGE_ALL_FILTERS_BUTTON_CLICK);
    };

    private handleCloseFilters = (): void => {
        this.setState({
            canRenderFiltersGroups: false,
        });

        this.handleBaseCloseFilters();
    };

    /* Handlers -> PriceFilter */

    private handleChangePriceFilter = (
        payload: IChangePriceFilterPayload,
        targetFiltersType: 'CURRENT' | 'ALL',
    ): void => {
        const {onChangePriceFilter} = this.props;

        onChangePriceFilter({
            ...payload,
            targetFiltersType,
        });

        reachGoal(EHotelsGoal.SEARCH_PAGE_PRICE_FILTER_CHANGE);
    };

    /* Handlers -> FilterGroup */

    private handleChangeFilterGroup = (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ): void => {
        const {onChangeFilterGroup} = this.props;

        onChangeFilterGroup({
            ...payload,
            targetFiltersType: 'CURRENT',
        });

        const filterChanges = {
            [meta.filterId]: meta.itemId,
        };

        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTER_GROUP_CHANGE, {
            hotels: {
                filters:
                    payload.atomsOn?.length || meta.filterId === 'ratings'
                        ? {apply: filterChanges}
                        : {unset: filterChanges},
            },
        });
    };

    /* Handlers -> QuickFilter */

    private handleChangeQuickFilter = (
        payload: IChangeFilterGroupPayload,
    ): void => {
        const {onChangeFilterGroup} = this.props;

        onChangeFilterGroup({
            ...payload,
            targetFiltersType: 'ALL',
        });

        reachGoal(EHotelsGoal.SEARCH_PAGE_QUICK_FILTERS_CHANGE);
    };

    /* Actions */

    private revertFilters = (): void => {
        const {onRevertFilters} = this.props;

        onRevertFilters();
    };

    private handleResetFilters = (): void => {
        const {onResetFilters} = this.props;

        onResetFilters();
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_RESET);
    };

    private applyFilters = (): void => {
        const {onApplyFilters} = this.props;

        onApplyFilters();
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_APPLY);
    };

    private hideFiltersGroups = (): void => {
        this.setState({canRenderFiltersGroups: false});
    };

    /* Render FiltersBar */

    private renderFiltersBar(): React.ReactNode {
        const {
            deviceInfo,
            mapListRadioButtonNode,
            filters: {detailedFiltersBatches},
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        if (!detailedFiltersBatches) {
            if (deviceInfo.isDesktop) {
                return (
                    <HotelsFiltersBarSkeleton
                        className={cx('filtersBarSkeleton')}
                    />
                );
            }

            if (hotelsHeaderRedesign) {
                return (
                    <MobileHotelsFiltersBarRedesignSkeleton
                        className={cx('filtersBarSkeletonRedesign')}
                    />
                );
            }

            return (
                <MobileHotelsFiltersBarSkeleton
                    className={cx('filtersBarSkeleton_mobile')}
                />
            );
        }

        return (
            <HotelsFiltersBar
                deviceInfo={deviceInfo}
                beforeFiltersNode={mapListRadioButtonNode}
                quickFiltersNode={this.renderQuickFiltersNode()}
                toggleFiltersVisibilityNode={this.renderToggleFiltersVisibilityButton()}
                hotelsHeaderRedesign={hotelsHeaderRedesign}
            />
        );
    }

    private renderPriceRangeFilter(
        targetFiltersType: 'CURRENT' | 'ALL',
    ): React.ReactNode {
        const {
            filters: {priceFilter},
            nights,
            deviceInfo,
        } = this.props;

        if (!priceFilter) {
            return null;
        }

        const componentNode = deviceInfo.isDesktop ? (
            <HotelsPriceFilter
                className={cx('priceFilter', 'priceFilter_shortView')}
                deviceType={deviceInfo}
                nights={nights}
                targetFiltersType={targetFiltersType}
                onChange={this.handleChangePriceFilter}
                {...priceFilter}
            />
        ) : (
            <HotelsPriceFilterWithInputs
                className={cx('priceFilter')}
                deviceType={deviceInfo}
                targetFiltersType={targetFiltersType}
                onChange={this.handleChangePriceFilter}
                {...priceFilter}
            />
        );

        return (
            <div className={cx(deviceMods('priceFilterContainer', deviceInfo))}>
                {componentNode}
            </div>
        );
    }

    private renderDesktopQuickFilters(): React.ReactNode {
        const {
            filters: {quickFilters},
        } = this.props;

        return this.renderQuickFilters(quickFilters);
    }

    private renderMobileQuickFilters(): React.ReactNode {
        const {
            filters: {quickFilters, quickControls},
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        if (hotelsHeaderRedesign && quickControls?.length) {
            return this.renderQuickFilters(quickControls);
        }

        if (!quickFilters?.length) return null;

        return this.renderQuickFilters(quickFilters);
    }

    private renderQuickFilters(
        quickFilters: IQuickControl[] | undefined,
    ): React.ReactNode {
        const {
            deviceInfo,
            renderSort,
            filters: {activeFilterAtoms},
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        if (quickFilters?.length) {
            return (
                <HotelsQuickFilters
                    className={cx(
                        'quickFilters',
                        deviceMods('quickFilters', deviceInfo),
                        hotelsHeaderRedesign && 'quickFilters_mobileRedesign',
                    )}
                    deviceType={deviceInfo}
                    items={quickFilters}
                    activeFilterAtoms={activeFilterAtoms}
                    hotelsHeaderRedesign={hotelsHeaderRedesign}
                    spaceBetweenButtons={2}
                    renderSort={renderSort}
                    onChange={this.handleChangeQuickFilter}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'quickFilters',
                    })}
                />
            );
        }

        return null;
    }

    private renderQuickFiltersNode(): React.ReactNode {
        const {
            deviceInfo: {isMobile},
        } = this.props;

        if (isMobile) {
            return this.renderMobileQuickFilters();
        }

        return (
            <div className={cx('filtersBarControls')}>
                {this.renderDesktopQuickFilters()}
                {this.renderPriceRangeFilter('ALL')}
            </div>
        );
    }

    private renderToggleFiltersVisibilityButton(): React.ReactNode {
        const {canRenderFiltersGroups} = this.state;
        const {
            deviceInfo,
            totalActiveFilters,
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        return (
            <ToggleFiltersVisibilityButton
                deviceInfo={deviceInfo}
                totalActiveFilters={totalActiveFilters}
                canRenderFiltersGroups={canRenderFiltersGroups}
                hotelsHeaderRedesign={hotelsHeaderRedesign}
                onToggleViewFilters={this.handleToggleViewFilters}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'allFiltersButton',
                })}
            />
        );
    }

    private renderCloseFiltersButton(): React.ReactNode {
        return (
            <div className={cx('closeWrapper')}>
                <button
                    className={cx('closeButton')}
                    onClick={this.handleCloseFilters}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'closeFiltersButton',
                    })}
                >
                    {closeFilters()} <CloseIcon className={cx('closeIcon')} />
                </button>
            </div>
        );
    }

    /* Render DetailedFilters Containers */

    private renderMobileHotelsFiltersContainer(): React.ReactNode {
        const {canRenderFiltersGroups} = this.state;

        return (
            <MobileHotelsFiltersContainer
                isVisible={canRenderFiltersGroups}
                onClose={this.handleCloseMobileFilters}
            >
                <>
                    {this.renderFiltersGroups()}
                    {this.renderControlButtons()}
                </>
            </MobileHotelsFiltersContainer>
        );
    }

    private renderDesktopHotelsFiltersContainer(): React.ReactNode {
        const {canDisableFilters, deviceInfo} = this.props;
        const {lockScroll, canRenderFiltersGroups, filtersContainerOffset} =
            this.state;
        const filterGroups = this.renderFiltersGroups();

        if (!filterGroups) {
            return null;
        }

        return (
            <DesktopHotelsFiltersContainer
                className={cx('allFiltersContainer')}
                deviceType={deviceInfo}
                isVisible={canRenderFiltersGroups}
                topOffset={filtersContainerOffset}
                canDisableFilters={canDisableFilters}
                onClose={this.hideFiltersGroups}
                onFinishAnimate={this.handleFinishAnimateDesktopFiltersGroups}
            >
                {this.renderCloseFiltersButton()}
                {filterGroups}

                <div className={cx('controlButtonsWrapper')}>
                    <Separator className={cx('separator')} />
                    {this.renderControlButtons()}
                </div>

                {lockScroll && <HideBodyVerticalScroll />}
            </DesktopHotelsFiltersContainer>
        );
    }

    /* Render DetailedFilters */

    private renderFiltersGroups(): React.ReactNode {
        const {
            filters: {detailedFiltersBatches, activeFilterAtoms},
            deviceInfo,
            nights,
        } = this.props;
        const {canRenderFiltersGroups} = this.state;

        if (detailedFiltersBatches) {
            return (
                <HotelsFiltersGroups
                    priceFilterNode={this.renderPriceRangeFilter('CURRENT')}
                    filterGroups={detailedFiltersBatches}
                    activeFilterAtoms={activeFilterAtoms}
                    isVisible={canRenderFiltersGroups}
                    deviceType={deviceInfo}
                    nights={nights}
                    onChangeFilterGroup={this.handleChangeFilterGroup}
                />
            );
        }

        return null;
    }

    private renderControlButtons(): React.ReactNode {
        const {
            deviceInfo,
            totalActiveFilters,
            filters: {foundHotelCount},
        } = this.props;

        return (
            <HotelsFiltersControlButtons
                className={cx(
                    'controlButtons',
                    deviceMods('controlButtons', deviceInfo),
                )}
                onResetFilters={this.handleResetFilters}
                onApplyFilters={this.handleApplyFiltersGroups}
                deviceInfo={deviceInfo}
                foundHotelCount={foundHotelCount}
                isDisabledResetButton={totalActiveFilters === 0}
            />
        );
    }

    render(): React.ReactNode {
        const {
            deviceInfo,
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        return (
            <div
                className={cx(
                    deviceModMobile('filters', deviceInfo),
                    hotelsHeaderRedesign && 'filters_redesign',
                )}
                ref={this.rootRef}
            >
                {this.renderFiltersBar()}
                {deviceInfo.isMobile
                    ? this.renderMobileHotelsFiltersContainer()
                    : this.renderDesktopHotelsFiltersContainer()}
            </div>
        );
    }
}

export default connect(hotelsFiltersSelector)(HotelsFilters);
