import {Component, ChangeEvent, ReactNode} from 'react';
import _find from 'lodash/find';

import {IWithClassName} from 'types/withClassName';

import Radiobox from 'components/Radiobox/Radiobox';

/* Init styles */
import cx from './BaseRadioBox.scss';

type WithId<OptionId> = {id: Extract<OptionId, string>};
type Option<OptionType, OptionId> = OptionType & WithId<OptionId>;

/* Components */
interface IBaseRadioBoxProps<OptionType, OptionId> extends IWithClassName {
    radioClassName?: string;
    options: Option<OptionType, OptionId>[];
    renderRadioLabel: (option: OptionType) => ReactNode;
    getRadioDisabled: (option: OptionType) => boolean;
    value?: Extract<OptionId, string>;
    onChange: (payload: {option: OptionType}) => void;
}

class BaseRadioBox<OptionType, OptionId> extends Component<
    IBaseRadioBoxProps<OptionType, OptionId>
> {
    /* Handlers */
    private handleChange = (e: ChangeEvent<HTMLInputElement>): void => {
        const {
            target: {value: optionId},
        } = e;
        const {onChange, options} = this.props;
        const option = _find<Option<OptionType, OptionId>>(
            options,
            ({id}: WithId<OptionId>) => String(id) === String(optionId),
        );

        if (option) {
            onChange({option});
        }
    };

    private mapOption = (option: Option<OptionType, OptionId>) => {
        const {getRadioDisabled, renderRadioLabel, radioClassName} = this.props;
        const disabled = getRadioDisabled(option);
        const {id} = option;

        return {
            value: id,
            label: renderRadioLabel(option) as any,
            className: cx(radioClassName, 'radio'),
            disabled,
        };
    };

    render() {
        const {options, className, value} = this.props;

        return (
            <Radiobox
                className={className}
                value={value}
                onChange={this.handleChange}
                options={options.map(this.mapOption)}
            />
        );
    }
}

export default BaseRadioBox;
