import {FC, MouseEvent, useRef, useEffect, useCallback, useState} from 'react';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {reachGoal} from 'utilities/metrika';

import cx from './DesktopHotelsFiltersContainer.scss';

/* Component Types */
interface IHotelsFiltersContainerProps extends IWithClassName, IWithDeviceType {
    isVisible: boolean;
    topOffset: number;
    canDisableFilters: boolean;
    onFinishAnimate?: () => void;
    onClose: () => void;
}

const HotelsFiltersContainer: FC<IHotelsFiltersContainerProps> = props => {
    const {className, isVisible, onFinishAnimate, deviceType} = props;
    const filtersGroupsRef = useRef<HTMLDivElement | null>(null);
    const [canUseMoveOutAnimation, setCanUseMoveOutAnimation] = useState(false);

    const handleLayoutClick = (e: MouseEvent<HTMLDivElement>): void => {
        const {onClose} = props;
        const targetNode = e.target;
        const filtersGroupsNode = filtersGroupsRef.current;

        if (targetNode && filtersGroupsNode) {
            const isFiltersGroupsClick = filtersGroupsNode.contains(
                targetNode as Node,
            );

            if (!isFiltersGroupsClick) {
                onClose();
                reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_LAYOUT_CLICK);
            }
        }
    };

    const handleTransitionEnd = useCallback(() => {
        if (!isVisible) {
            setCanUseMoveOutAnimation(false);

            if (onFinishAnimate) {
                onFinishAnimate();
            }
        }
    }, [isVisible, onFinishAnimate]);

    useEffect(() => {
        if (isVisible) {
            setCanUseMoveOutAnimation(true);
        }
    }, [isVisible]);

    return (
        <div
            className={cx(
                'filtersGroupsContainer',
                {
                    filtersGroupsContainer_moveIn: isVisible,
                    filtersGroupsContainer_moveOut:
                        canUseMoveOutAnimation && !isVisible,
                },
                className,
            )}
            style={{top: props.topOffset}}
        >
            <div
                className={cx('filtersGroupsOverlay')}
                onClick={handleLayoutClick}
                onTransitionEnd={handleTransitionEnd}
            />
            <div
                className={cx('filtersGroups', {
                    filtersGroups_ieFix: deviceType.isIe,
                    filtersGroups_disable: props.canDisableFilters,
                })}
                ref={filtersGroupsRef}
            >
                {props.children}
            </div>
        </div>
    );
};

export default HotelsFiltersContainer;
