import {PureComponent, ReactNode} from 'react';
import {flatten} from 'lodash';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {
    ActiveFilterAtomsType,
    EDetailedFiltersBatchItemType,
    IBasicFilter,
    IBasicFilterGroup,
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
    IDetailedFiltersBatch,
    IDetailedFilterItem,
} from 'types/hotels/search/IFiltersInfo';
import {IWithDeviceType} from 'types/withDeviceType';

import scrollTo from 'utilities/dom/scrollTo';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import {priceDescription} from 'i18n/hotels-SearchPageFilters';

import HotelsCheckboxList from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/components/HotelsCheckboxList/HotelsCheckboxList';
import HotelsRadioBox from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/components/HotelsRadioBox/HotelsRadioBox';
import Rating from 'components/Rating/Rating';

import cx from './HotelsFiltersGroups.scss';

/* Component Types */
interface IHotelsFiltersGroupsProps extends IWithDeviceType {
    className?: string;
    filterGroups: IDetailedFiltersBatch[];
    onChangeFilterGroup: (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ) => void;
    nights: number;
    activeFilterAtoms: ActiveFilterAtomsType;
    priceFilterNode: ReactNode;
    isVisible: boolean;
}

class HotelsFiltersGroups extends PureComponent<IHotelsFiltersGroupsProps> {
    private scrollContainerNode: HTMLDivElement | null = null;

    componentDidMount(): void {
        if (this.scrollContainerNode) {
            this.scrollContainerNode.addEventListener(
                'scroll',
                this.handleScroll,
            );
        }
    }

    componentDidUpdate(prevProps: Readonly<IHotelsFiltersGroupsProps>): void {
        if (
            !prevProps.isVisible &&
            this.props.isVisible &&
            this.scrollContainerNode
        ) {
            scrollTo({top: 0}, this.scrollContainerNode);
        }
    }

    componentWillUnmount(): void {
        this.unsubscribe();
    }

    setScrollContainerNodeRef = (dayNode: HTMLDivElement | null): void => {
        this.scrollContainerNode = dayNode;
    };

    private unsubscribe(): void {
        if (this.scrollContainerNode) {
            this.scrollContainerNode.removeEventListener(
                'scroll',
                this.handleScroll,
            );
        }
    }

    private parseEffect(effect: string | undefined): number | null {
        if (effect?.includes('rating:')) {
            return parseInt(effect.replace('rating:', ''), 10);
        }

        return 0;
    }

    private handleScroll = (): void => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTER_GROUP_SCROLL);
        this.unsubscribe();
    };

    private renderFiltersGroup = (
        filterGroup: IBasicFilterGroup,
    ): ReactNode => {
        const {onChangeFilterGroup, activeFilterAtoms, deviceType} = this.props;
        const {items, type, name, id} = filterGroup;

        if (type) {
            switch (type) {
                case 'MULTI': {
                    return (
                        <HotelsCheckboxList
                            id={id}
                            key={name}
                            items={items}
                            deviceType={deviceType}
                            activeFilterAtoms={activeFilterAtoms}
                            onChange={onChangeFilterGroup}
                        />
                    );
                }

                case 'SINGLE': {
                    return (
                        <HotelsRadioBox
                            id={id}
                            key={name}
                            options={items}
                            deviceType={deviceType}
                            activeFilterAtoms={activeFilterAtoms}
                            onChange={onChangeFilterGroup}
                            renderRadioLabel={
                                id === 'ratings'
                                    ? this.renderRatingFilterItem
                                    : undefined
                            }
                        />
                    );
                }

                default: {
                    return null;
                }
            }
        } else {
            return null;
        }
    };

    private renderRatingFilterItem = (item: IBasicFilter): ReactNode => {
        return (
            <Rating
                size="s"
                className={cx('rating')}
                rating={this.parseEffect(item.effect)}
                text={item.name}
            />
        );
    };

    private renderFiltersGroups(): ReactNode {
        const {filterGroups, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'filtersGroups',
                    `filtersGroups_columns_${filterGroups.length}`,
                )}
            >
                {deviceType.isMobile && (
                    <>
                        {this.isWithoutPriceFilter() &&
                            this.renderFiltersColumn(
                                [
                                    {
                                        type: EDetailedFiltersBatchItemType.PRICE,
                                    },
                                ],
                                -1,
                            )}
                    </>
                )}

                {filterGroups.map((group, index) =>
                    this.renderFiltersColumn(group.items, index),
                )}
            </div>
        );
    }

    private renderFiltersColumn(
        filters: IDetailedFilterItem[],
        index: number,
    ): ReactNode {
        return (
            <div className={cx('column')} key={index}>
                {filters.map(({type, detailedFilters: filter}) => {
                    if (type === EDetailedFiltersBatchItemType.PRICE) {
                        return this.renderPriceFilter();
                    }

                    if (!filter) {
                        return null;
                    }

                    const filtersGroupNode = this.renderFiltersGroup(filter);

                    if (filtersGroupNode) {
                        return (
                            <div className={cx('group')} key={filter.name}>
                                <div className={cx('groupTitle')}>
                                    {filter.name}
                                </div>
                                {filtersGroupNode}
                            </div>
                        );
                    }

                    return null;
                })}
            </div>
        );
    }

    private renderPriceFilter(): ReactNode {
        const {deviceType, nights, priceFilterNode} = this.props;

        return (
            <div className={cx('group', 'group_price')} key="price">
                {Boolean(deviceType.isMobile && priceFilterNode && nights) && (
                    <div className={cx('groupTitle')}>
                        {priceDescription({
                            withPrefix: true,
                            nights,
                        })}
                    </div>
                )}

                {priceFilterNode}
            </div>
        );
    }

    private isWithoutPriceFilter(): boolean {
        const {filterGroups} = this.props;

        return !flatten(filterGroups.map(item => item.items)).find(
            item => item.type === EDetailedFiltersBatchItemType.PRICE,
        );
    }

    render(): ReactNode {
        const {deviceType} = this.props;

        return (
            <div
                className={cx(
                    'filtersGroupsContent',
                    deviceMods('filtersGroupsContent', deviceType),
                )}
                ref={this.setScrollContainerNodeRef}
            >
                {this.renderFiltersGroups()}
            </div>
        );
    }
}

export default HotelsFiltersGroups;
