import {useCallback, memo} from 'react';
import _isNumber from 'lodash/isNumber';

/* Types */
import {
    IRangePriceFilter,
    IChangePriceFilterPayload,
} from 'types/hotels/search/IFiltersInfo';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

/* Utilities */
import calculateStepByRangeValues from './utilities/calculateStepByRangeValues';
import {CHAR_SPACE} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/hotels-SearchPageFilters';

import InputRange from 'components/InputRange/InputRange';
import Price from 'components/Price/Price';
import Text from 'components/Text/Text';

/* Component Types */
interface IHotelsPriceFilterProps
    extends IWithClassName,
        IRangePriceFilter,
        IWithDeviceType {
    nights: number;
    canShowNights?: boolean;
    targetFiltersType: 'CURRENT' | 'ALL';
    onChange: (
        payload: IChangePriceFilterPayload,
        targetFiltersType: 'CURRENT' | 'ALL',
    ) => void;
}

const HotelsPriceFilterComponent = memo(function HotelsPriceFilter(
    props: IHotelsPriceFilterProps,
) {
    const {
        className,
        currency,
        nights,
        canShowNights,
        minPriceEstimate,
        maxPriceEstimate,
        minValue,
        maxValue,
        deviceType: {isMobile},
        onChange,
        targetFiltersType,
    } = props;

    const renderHandleTooltip = useCallback(
        ({value, alignment}: {value: number; alignment: 'left' | 'right'}) => (
            <Text size="s">
                <Price
                    value={value}
                    isFrom={alignment === 'left'}
                    isUpTo={alignment === 'right'}
                    postfix={value === maxPriceEstimate ? '+' : ''}
                    currency={currency}
                />
                {canShowNights &&
                    alignment === 'right' &&
                    `${CHAR_SPACE}${i18nBlock.priceDescription({
                        nights,
                        withPrefix: false,
                    })}`}
            </Text>
        ),
        [maxPriceEstimate, currency, nights, canShowNights],
    );

    const handleChange = useCallback(
        (changeParams: readonly number[]): void => {
            const [changedMinValue, changedMaxValue] = changeParams;

            onChange(
                {
                    minValue: changedMinValue,
                    maxValue: changedMaxValue,
                },
                targetFiltersType,
            );
        },
        [onChange, targetFiltersType],
    );

    if (_isNumber(minValue) && _isNumber(maxValue)) {
        return (
            <InputRange
                className={className}
                rangeValues={[minPriceEstimate, maxPriceEstimate]}
                values={[minValue, maxValue]}
                onChange={handleChange}
                step={calculateStepByRangeValues({
                    minPriceEstimate,
                    maxPriceEstimate,
                })}
                tooltipPosition="top"
                size={isMobile ? 'm' : 's'}
                renderHandleTooltip={renderHandleTooltip}
            />
        );
    }

    return null;
});

export default HotelsPriceFilterComponent;
