import {useCallback, memo, useState, useRef, useMemo} from 'react';
import _isNumber from 'lodash/isNumber';
import throttle from 'lodash/throttle';

/* Types */

import {
    IRangePriceFilter,
    IChangePriceFilterPayload,
} from 'types/hotels/search/IFiltersInfo';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

/* Utilities */
import transformValues from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/components/HotelsPriceFilter/utilities/transformValues';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelsPriceFilterWithInputs';

import InputRange from 'components/InputRange/InputRange';
import PriceInput from 'components/PriceInput/PriceInput';

import cx from './HotelsPriceFilterWithInputs.scss';

const QA_PREFIX = 'hotelsPriceFilterWithInputs';

const THROTTLE_THRESHOLD = 100;

/* Component Types */
type TDraggedRangeValues = {
    minValue?: number;
    maxValue?: number;
} | null;

export interface IHotelsPriceFilterWithInputsProps
    extends IWithClassName,
        IRangePriceFilter,
        IWithDeviceType {
    targetFiltersType: 'CURRENT' | 'ALL';
    onChange: (
        payload: IChangePriceFilterPayload,
        targetFiltersType: 'CURRENT' | 'ALL',
    ) => void;
}

const HotelsPriceFilterWithInputs = memo(function HotelsPriceFilter(
    props: IHotelsPriceFilterWithInputsProps,
) {
    const {
        className,
        currency,
        minPriceEstimate,
        maxPriceEstimate,
        minValue,
        maxValue,
        deviceType: {isMobile},
        onChange,
        targetFiltersType,
    } = props;

    // Будем хранить здесь значения в тот момент, когда ручку дергают мышкой,
    // Чтобы не плодить запросы на каждый чих
    const [draggedRangeValues, setDraggedRangeValues] =
        useState<TDraggedRangeValues>(null);

    const inputContainers = useRef<HTMLDivElement | null>(null);

    const handleOutsideInteraction = useCallback(() => {
        const activeElement =
            typeof document !== 'undefined' &&
            (document.activeElement as HTMLInputElement);
        const container = inputContainers.current;

        if (
            !activeElement ||
            !container ||
            !container.contains(activeElement)
        ) {
            return;
        }

        activeElement.blur?.();
    }, []);

    const handleChange = useCallback(
        (changeParams: readonly number[]): void => {
            const [changedMinValue, changedMaxValue] = changeParams;

            onChange(
                {
                    minValue: changedMinValue,
                    maxValue: changedMaxValue,
                },
                targetFiltersType,
            );

            setDraggedRangeValues(null);
        },
        [onChange, targetFiltersType],
    );

    const handleSliderChange = useCallback(
        (changeParams: readonly number[]): void => {
            const roundedValues = transformValues(
                changeParams,
                minValue ?? minPriceEstimate,
                maxValue ?? maxPriceEstimate,
                {minPriceEstimate, maxPriceEstimate},
            );

            handleChange(roundedValues);
        },
        [handleChange, minValue, maxValue, minPriceEstimate, maxPriceEstimate],
    );

    const handleSliderUpdate = useCallback(
        (changeParams: readonly number[]): void => {
            handleOutsideInteraction();

            const roundedValues = transformValues(
                changeParams,
                minValue ?? minPriceEstimate,
                maxValue ?? maxPriceEstimate,
                {minPriceEstimate, maxPriceEstimate},
            );

            setDraggedRangeValues({
                minValue: roundedValues[0],
                maxValue: roundedValues[1],
            });
        },
        [
            minValue,
            maxValue,
            maxPriceEstimate,
            minPriceEstimate,
            handleOutsideInteraction,
        ],
    );

    const handleSliderUpdateThrottled = useMemo(
        () => throttle(handleSliderUpdate, THROTTLE_THRESHOLD),
        [handleSliderUpdate],
    );

    const handleMinInputChange = useCallback(
        (newValue: number) => {
            handleChange([newValue, maxValue ?? maxPriceEstimate]);
        },
        [maxValue, maxPriceEstimate, handleChange],
    );

    const handleMaxInputChange = useCallback(
        (newValue: number) => {
            handleChange([minValue ?? minPriceEstimate, newValue]);
        },
        [minValue, minPriceEstimate, handleChange],
    );

    if (!_isNumber(minValue) || !_isNumber(maxValue)) {
        return null;
    }

    const displayMin = draggedRangeValues?.minValue ?? minValue;
    const displayMax = draggedRangeValues?.maxValue ?? maxValue;

    return (
        <div>
            <div className={cx('inputs')} ref={inputContainers}>
                <PriceInput
                    value={displayMin}
                    currency={currency}
                    onValueChange={handleMinInputChange}
                    beforeText={i18nBlock._from()}
                    {...prepareQaAttributes({
                        parent: 'hotelsPriceFilterWithInputs',
                        current: 'from',
                    })}
                />
                <PriceInput
                    className={cx('input', 'input_last')}
                    value={displayMax}
                    currency={currency}
                    onValueChange={handleMaxInputChange}
                    postfix={displayMax === maxPriceEstimate ? '+' : ''}
                    beforeText={i18nBlock.to()}
                    {...prepareQaAttributes({
                        parent: QA_PREFIX,
                        current: 'to',
                    })}
                />
            </div>
            <InputRange
                className={className}
                rangeValues={[minPriceEstimate, maxPriceEstimate]}
                values={[displayMin, displayMax]}
                onChange={handleSliderChange}
                onUpdate={handleSliderUpdateThrottled}
                canRenderTooltip={false}
                size={isMobile ? 'm' : 's'}
                {...prepareQaAttributes({parent: QA_PREFIX, current: 'range'})}
            />
        </div>
    );
});

export default HotelsPriceFilterWithInputs;
