import {FunctionComponent, memo, useMemo, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    ActiveFilterAtomsType,
    EQuickFilterEffect,
    TQuickFilter,
} from 'types/hotels/search/IFiltersInfo';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useWindowLayout} from 'utilities/hooks/useWindowLayout';
import {deviceModMobile} from 'utilities/stylesUtils';
import getIconProps from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/components/HotelsQuickFilter/utilities/getIconProps';
import getHintContent from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsFilters/components/HotelsQuickFilter/utilities/getHintContent';

/* Components */
import CheckButton from 'components/CheckButton/CheckButton';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import cx from './HotelsQuickFilter.scss';

interface IHotelsQuickMirFilterProps extends IWithClassName {
    item: TQuickFilter;
    activeFilterAtoms: ActiveFilterAtomsType;
    handleCheckButtonClick: () => void;
    isChecked: boolean;
    hotelsHeaderRedesign?: boolean;
}

const SMALL_SCREEN_THRESHOLD = 360;

const HotelsQuickFilter: FunctionComponent<IHotelsQuickMirFilterProps> = ({
    item,
    handleCheckButtonClick,
    isChecked,
    className,
    hotelsHeaderRedesign,
}) => {
    const {
        value: isHintVisible,
        setFalse: hideHint,
        setTrue: showHint,
        toggle: toggleHintVisibility,
    } = useBoolean(false);

    const buttonRef = useRef<HTMLDivElement>(null);

    const {id, enabled, name, effect} = item;
    const deviceType = useDeviceType();
    const {isDesktop, isMobile} = deviceType;
    const {width} = useWindowLayout();

    const text = useMemo(() => {
        const isSmallScreen = width < SMALL_SCREEN_THRESHOLD;

        if (effect === EQuickFilterEffect.MIR_CASHBACK && isSmallScreen) {
            return;
        }

        return name;
    }, [name, width, effect]);

    const iconProps = useMemo(
        () => getIconProps(item, deviceType),
        [item, deviceType],
    );

    const hintNode = useMemo(
        () => getHintContent(item, deviceType),
        [item, deviceType],
    );

    const button = (
        <CheckButton
            key={id}
            className={cx(
                'filter',
                {
                    filter_redesign: hotelsHeaderRedesign,
                },
                effect && `filter__${effect}`,
                !enabled && `filter__disabled`,
                deviceModMobile('filter', deviceType),
                className,
            )}
            checked={isChecked}
            size={isDesktop || hotelsHeaderRedesign ? 'm' : 'l'}
            theme={isDesktop || hotelsHeaderRedesign ? 'secondary' : 'outlined'}
            disabled={!enabled}
            onClick={handleCheckButtonClick}
            {...iconProps}
        >
            {text}
        </CheckButton>
    );

    return enabled || !hintNode ? (
        button
    ) : (
        <>
            <div
                ref={buttonRef}
                onMouseEnter={isDesktop ? showHint : undefined}
                onMouseLeave={isDesktop ? hideHint : undefined}
                onClick={isMobile ? toggleHintVisibility : undefined}
            >
                {button}
            </div>

            <MessageBoxPopupOrBottomSheet
                isVisible={isHintVisible}
                onClose={hideHint}
                messageBoxProps={{
                    className: cx('messageBox'),
                    anchorRef: buttonRef,
                    direction: EPopupDirection.BOTTOM,
                }}
            >
                <div
                    className={cx(
                        'toggleContent',
                        deviceModMobile('toggleContent', deviceType),
                    )}
                >
                    {hintNode}
                </div>
            </MessageBoxPopupOrBottomSheet>
        </>
    );
};

export default memo(HotelsQuickFilter);
