import {PureComponent, ReactNode} from 'react';

/* Types */
import {
    ActiveFilterAtomsType,
    IQuickControl,
    TQuickFilter,
    IChangeFilterGroupPayload,
    EQuickControlType,
} from 'types/hotels/search/IFiltersInfo';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile} from 'utilities/stylesUtils';

/* Components */
import {TBoxSizes} from 'components/Box/Box';
import Flex from 'components/Flex/Flex';

import HotelsQuickFilter from '../HotelsQuickFilter/HotelsQuickFilter';

/* Init styles */
import cx from './HotelsQuickFilters.scss';

/* Component Types */
interface IHotelsQuickFiltersProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    items: IQuickControl[];
    activeFilterAtoms: ActiveFilterAtomsType;
    spaceBetweenButtons: TBoxSizes;
    filterClassName?: string;
    hotelsHeaderRedesign?: boolean;
    onChange(payload: IChangeFilterGroupPayload): void;
    renderSort?: () => ReactNode;
}

class HotelsQuickFilters extends PureComponent<IHotelsQuickFiltersProps> {
    private handleCheckButtonClick = (item: TQuickFilter) => (): void => {
        const {onChange} = this.props;
        const {atomsOn, atomsOff} = item;
        const isChecked = this.getQuickFilterChecked(item);

        onChange(isChecked ? {atomsOff: atomsOn} : {atomsOn, atomsOff});
    };

    private getQuickFilterChecked = (item: TQuickFilter): boolean => {
        const {activeFilterAtoms} = this.props;
        const {atomsOn, atomsOff} = item;

        return (
            atomsOn.every(atom => activeFilterAtoms[atom]) &&
            atomsOff.every(atom => !activeFilterAtoms[atom])
        );
    };

    private renderQuickControl = (item: IQuickControl): ReactNode => {
        const {renderSort} = this.props;

        if (!item.type || item.type === EQuickControlType.QUICK_FILTER) {
            return this.renderQuickFilter(item as TQuickFilter);
        }

        if (
            item.type === EQuickControlType.QUICK_SORT &&
            typeof renderSort === 'function'
        ) {
            return renderSort();
        }

        return null;
    };

    private renderQuickFilter = (item: TQuickFilter): ReactNode => {
        const {deviceType, filterClassName, hotelsHeaderRedesign} = this.props;
        const {id} = item;
        const filterButtonClassName = cx(
            'filter',
            filterClassName,
            deviceModMobile('filter', deviceType),
            hotelsHeaderRedesign && 'filter_redesign',
        );

        return (
            <HotelsQuickFilter
                key={id}
                item={item}
                activeFilterAtoms={this.props.activeFilterAtoms}
                handleCheckButtonClick={this.handleCheckButtonClick(item)}
                className={filterButtonClassName}
                isChecked={this.getQuickFilterChecked(item)}
                hotelsHeaderRedesign={hotelsHeaderRedesign}
            />
        );
    };

    private renderItems(): ReactNode {
        const {items} = this.props;

        return items.map(this.renderQuickControl);
    }

    render(): ReactNode {
        const {className, spaceBetweenButtons} = this.props;
        const items = this.renderItems();

        return (
            <Flex
                className={className}
                inline
                between={spaceBetweenButtons}
                flexWrap="nowrap"
                {...prepareQaAttributes(this.props)}
            >
                {items}
            </Flex>
        );
    }
}

export default HotelsQuickFilters;
