import React, {PureComponent, ReactNode} from 'react';
import _find from 'lodash/find';

/* Types */
import {
    IBasicFilter,
    FilterAtomType,
    ActiveFilterAtomsType,
    IChangeFilterGroupPayload,
    IChangeFilterGroupMeta,
} from 'types/hotels/search/IFiltersInfo';
import {IWithDeviceType} from 'types/withDeviceType';

import {deviceMods} from 'utilities/stylesUtils';

/* Components */
import BaseRadioBox from '../BaseRadioBox/BaseRadioBox';

/* Component Types */
interface IHotelsRadioBoxProps extends IWithDeviceType {
    id: string;
    options: IBasicFilter[];
    activeFilterAtoms: ActiveFilterAtomsType;
    renderRadioLabel?: (option: IBasicFilter) => ReactNode;
    onChange: (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ) => void;
}

/* Init styles */
import cx from './HotelsRadioBox.scss';

class HotelsRadioBox extends PureComponent<IHotelsRadioBoxProps> {
    /* Handlers */
    private handleChangeRadioBox = ({
        option: changedOption,
    }: {
        option: IBasicFilter;
    }): void => {
        const {onChange, options, id} = this.props;
        const {atoms} = changedOption;

        const atomsOff = options.reduce(
            (resultAtoms: FilterAtomType[], option) => {
                if (changedOption === option) {
                    return resultAtoms;
                }

                return [...resultAtoms, ...option.atoms];
            },
            [],
        );

        onChange(
            {atomsOff, atomsOn: atoms},
            {
                filterId: id,
                itemId: changedOption.id,
            },
        );
    };

    /* Helpers */
    private checkActiveOption = ({atoms}: IBasicFilter): boolean => {
        const {activeFilterAtoms} = this.props;

        return (
            atoms.length !== 0 && atoms.every(atom => activeFilterAtoms[atom])
        );
    };

    private getRadioBoxValue = (): string | undefined => {
        const {options} = this.props;
        const optionsWithAtoms = options.filter(({atoms}) => atoms.length);
        const activeOption = _find(optionsWithAtoms, this.checkActiveOption);

        if (activeOption) {
            return activeOption.id;
        }

        const optionWithoutAtoms = options.find(({atoms}) => !atoms.length);

        if (optionWithoutAtoms) {
            return optionWithoutAtoms.id;
        }
    };

    getRadioDisabled = (option: IBasicFilter): boolean => !option.enabled;

    /* Render */
    renderRadioLabel = (option: IBasicFilter): React.ReactNode => {
        const {renderRadioLabel} = this.props;
        const {name, hint} = option;

        return (
            <div className={cx('radioLabel')}>
                <div className={cx('radioName')}>
                    {renderRadioLabel ? renderRadioLabel(option) : name}
                </div>
                {Boolean(hint) && <div className={cx('radioHint')}>{hint}</div>}
            </div>
        );
    };

    render(): ReactNode {
        const {options, deviceType} = this.props;

        return (
            <BaseRadioBox<IBasicFilter, string>
                className={cx('group', deviceMods('group', deviceType))}
                radioClassName={cx('radio')}
                options={options}
                value={this.getRadioBoxValue()}
                getRadioDisabled={this.getRadioDisabled}
                renderRadioLabel={this.renderRadioLabel}
                onChange={this.handleChangeRadioBox}
            />
        );
    }
}

export default HotelsRadioBox;
