import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {getSearchInfo} from 'selectors/depreacted/hotels/search/searchHotels/getSearchInfo';

import humanizePeriod from 'utilities/dateUtils/humanizePeriod';
import {
    formatCountOfChildren,
    formatCountOfAdults,
} from 'utilities/formatCount';
import formatInteger from 'utilities/numbers/formatInteger';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-SearchInformationSection';

import cx from './HotelsSearchInformationSection.scss';

export interface IHotelsSearchInformationSectionProps
    extends IWithClassName,
        IWithQaAttributes {}

const HotelsSearchInformationSection: React.FC<IHotelsSearchInformationSectionProps> =
    props => {
        const {className} = props;

        const {data, isLoading} = useSelector(getSearchInfo);

        if (!data || isLoading) {
            return (
                <div className={cx('searchInformationSkeleton', className)} />
            );
        }

        const {
            offerSearchParams: {
                adults,
                checkinDate,
                checkoutDate,
                childrenAges: {length: childrens},
            },
            actualRegion: {name: regionName},
            foundHotelCount,
        } = data;

        const adultsLabel = formatCountOfAdults(adults);
        const datesLabel = humanizePeriod(checkinDate, checkoutDate);
        const childrenLabel =
            childrens > 0 ? formatCountOfChildren(childrens) : undefined;
        const foundHotelCountLabel = formatInteger(foundHotelCount);

        const totalLabel = i18nBlock.foundHotelNew({
            foundHotelCount,
            foundHotelCountLabel,
        });

        return (
            <section className={cx('searchInformation', className)}>
                <span
                    className={cx('totalLabel')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'totalLabel',
                    })}
                >
                    {totalLabel}.
                </span>

                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'searchParams',
                    })}
                >
                    {regionName}, {datesLabel}, {adultsLabel}
                    {childrenLabel ? `, ${childrenLabel}` : ''}
                </span>
            </section>
        );
    };

export default HotelsSearchInformationSection;
