import React, {ChangeEvent, PureComponent} from 'react';
import {IYMapsApi, IYMapsInstance} from 'react-yandex-maps';
import {isEqual} from 'lodash';
import {connect} from 'react-redux';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';
import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {MapBoundsType} from 'types/common/ICoordinates';
import {EPortalSeedReferer} from 'types/hotels/common/IAttribution';
import {
    EActiveHotelSource,
    IActiveHotel,
} from 'types/hotels/hotel/IActiveHotel';
import {IWithClassName} from 'types/withClassName';

import {searchMapSelector} from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsSearchMap/selectors/searchMapSelector';

import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    getSearchParamsForBaseHotelUrl,
    hotelsURLs,
} from 'projects/depreacted/hotels/utilities/urls';

import * as i18nBlock from 'i18n/hotels-HotelsMap';

/* Components */
import YandexMaps from 'components/YandexMaps/YandexMaps';
import Button from 'components/Button/Button';
import Checkbox from 'components/Checkbox/Checkbox';
import HotelsMapMarker from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsSearchMap/components/HotelsMapMarker/HotelsMapMarker';
import MapHotelCard from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsSearchMap/components/MapHotelCard/MapHotelCard';
import MobileMapHotelCard from 'projects/depreacted/hotels/pages/SearchPage/components/HotelsSearchMap/components/MobileMapHotelCard/MobileMapHotelCard';
import NoResultsMessage from './components/NoResultsMessage/NoResultsMessage';
import Spinner from 'components/Spinner/Spinner';

/* Init styles */

import cx from './HotelsSearchMap.scss';

interface IHotelsSearchMapProps extends IWithClassName, IWithQaAttributes {
    mapClassName?: string;
    isFullViewMap: boolean;
    mapHeight?: string | number;
    mapWidth?: string | number;

    onFavoriteClick?({
        permalink,
        isFavorite,
    }: {
        permalink: PermalinkType;
        isFavorite: boolean;
    }): void;
    onLoadMap?(): void;
    onLeaveMapMarker?(): void;
    onHoverMapMarker?(permalink: IActiveHotel | undefined): void;
    onClickMapMarker?(permalink: PermalinkType): void;
    onBoundsChange?(bounds: MapBoundsType): void;
    onResetSelectedCard?(): void;
}

interface IHotelsSearchMapState {
    mapSuccessLoaded: boolean;
    canRenderSearchButton: boolean;
    canStartSearchAfterBoundsChange: boolean;
    activeMarkerPosition: Nullable<IActiveMarkerPosition>;
    bounds?: MapBoundsType;
    containerSize: {
        width: number;
        height: number;
    };
}

interface IActiveMarkerPosition {
    left: number;
    top: number;
    bottom: number;
}

type THotelsSearchMapProps = IHotelsSearchMapProps &
    ReturnType<typeof searchMapSelector>;

/* Constants */
const MAP_CONTROLS: string[] = [];

class HotelsSearchMap extends PureComponent<
    THotelsSearchMapProps,
    IHotelsSearchMapState
> {
    state: IHotelsSearchMapState = {
        mapSuccessLoaded: false,
        canRenderSearchButton: false,
        activeMarkerPosition: null,
        canStartSearchAfterBoundsChange: true,
        containerSize: {
            width: 0,
            height: 0,
        },
    };

    private windowResized: boolean = false;

    private yandexMapAPIInstance: IYMapsApi | null = null;

    private yandexMapInstance: IYMapsInstance | null = null;

    private mapActionFinished: boolean = true;

    componentDidUpdate(prevProps: THotelsSearchMapProps): void {
        this.checkUpdateContainerSize(prevProps);
        this.checkUpdateBboxAsStruct(prevProps);
    }

    /* Check didUpdate helpers */

    private checkUpdateContainerSize = (
        prevProps: THotelsSearchMapProps,
    ): void => {
        const {mapHeight, mapWidth, isFullViewMap} = this.props;

        if (
            prevProps.mapHeight === mapHeight &&
            prevProps.mapWidth === mapWidth
        ) {
            return;
        }

        if (prevProps.isFullViewMap === isFullViewMap) {
            this.windowResized = true;
        }

        this.fitContainerToViewport();
        this.calculateContainerSize();
    };

    private checkUpdateBboxAsStruct = (
        prevProps: THotelsSearchMapProps,
    ): void => {
        const {bboxAsStruct} = this.props;

        if (prevProps.bboxAsStruct !== bboxAsStruct) {
            this.setState({bounds: bboxAsStruct});
        }
    };

    /* Helpers */

    private canRenderMarkerByMapBounds = (
        hotelWithOffers: IHotelWithOffers,
    ): boolean => {
        const {bounds} = this.state;
        const {
            hotel: {
                coordinates: {lat, lon},
            },
        } = hotelWithOffers;

        if (bounds) {
            const [leftBottomPoint, rightTopPoint] = bounds;

            const yandexMapAPIInstance = this.yandexMapAPIInstance;

            if (!yandexMapAPIInstance) {
                return true;
            }

            /**
             * @link https://yandex.ru/dev/maps/jsapi/doc/2.1/ref/reference/util.bounds.html#method_detail__containsPoint
             */
            return yandexMapAPIInstance.util.bounds.containsPoint(
                [
                    [leftBottomPoint.lat, leftBottomPoint.lon],
                    [rightTopPoint.lat, rightTopPoint.lon],
                ],
                [lat, lon],
            );
        }

        return true;
    };

    /* Refs manipulate */

    private setYandexMapInstanceRef = (
        yandexMapInstance: IYMapsInstance,
    ): void => {
        this.yandexMapInstance = yandexMapInstance;
    };

    /* Actions */

    private fitContainerToViewport = (): void => {
        if (this.yandexMapInstance) {
            const yandexMapContainer = this.yandexMapInstance.container;

            yandexMapContainer.fitToViewport();
            this.setState({canRenderSearchButton: false});
        }
    };

    private calculateContainerSize = (): void => {
        if (this.yandexMapInstance) {
            const yandexMapContainer = this.yandexMapInstance.container;
            const [width, height] = yandexMapContainer.getSize() || [0, 0];

            this.setState({
                containerSize: {
                    width,
                    height,
                },
            });
        }
    };

    /* Handlers */

    private handleChangeActiveMarkerPosition = (
        markerPosition: IActiveMarkerPosition,
    ): void => {
        const {activeMarkerPosition} = this.state;

        if (
            !activeMarkerPosition ||
            !isEqual(activeMarkerPosition, markerPosition)
        ) {
            this.setState({
                activeMarkerPosition: markerPosition,
            });
        }
    };

    private handleLoadMap = (yandexMapAPIInstance: IYMapsApi): void => {
        if (yandexMapAPIInstance) {
            const {onLoadMap} = this.props;

            this.yandexMapAPIInstance = yandexMapAPIInstance;

            this.setState({mapSuccessLoaded: true});
            this.fitContainerToViewport();
            this.calculateContainerSize();

            if (onLoadMap) {
                onLoadMap();
            }
        }
    };

    private handleChangeSearchTypeByMapBounds = (
        e: ChangeEvent<HTMLInputElement>,
    ): void => {
        const {
            target: {checked},
        } = e;

        this.setState({
            canStartSearchAfterBoundsChange: checked,
        });
    };

    private handleMapActionBegin = (): void => {
        this.mapActionFinished = false;
    };

    private handleMapActionEnd = (): void => {
        this.mapActionFinished = true;
    };

    private handleHoverMapMarker = (permalink?: PermalinkType): void => {
        const {onHoverMapMarker, onLeaveMapMarker} = this.props;

        if (this.mapActionFinished) {
            if (permalink) {
                if (onHoverMapMarker) {
                    onHoverMapMarker({
                        permalink,
                        source: EActiveHotelSource.MAP,
                    });
                }
            } else if (onLeaveMapMarker) {
                onLeaveMapMarker();
            }
        }
    };

    private handleBoundsChange = (
        bounds: MapBoundsType,
        {isUserEvent}: {isUserEvent: boolean},
    ): void => {
        const {onBoundsChange} = this.props;
        const {canStartSearchAfterBoundsChange} = this.state;

        if (canStartSearchAfterBoundsChange) {
            if (onBoundsChange && !this.windowResized && isUserEvent) {
                onBoundsChange(bounds);
            }
        } else {
            this.setState({canRenderSearchButton: isUserEvent});
        }

        this.windowResized = false;
        this.setState({bounds});
    };

    private handleStartChangeZoom = (): void => {
        const {onHoverMapMarker, activeHotelPermalink} = this.props;

        if (onHoverMapMarker && activeHotelPermalink) {
            onHoverMapMarker(undefined);
        }

        reachGoal(EHotelsGoal.SEARCH_PAGE_MAP_ZOOM_CHANGE);
    };

    private handleChangeZoomByControlButton = (): void => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_MAP_ZOOM_CHANGE);
    };

    private handleSearchButtonClick = (): void => {
        const {bounds} = this.state;
        const {onBoundsChange} = this.props;

        if (bounds) {
            this.setState({canRenderSearchButton: false});

            if (onBoundsChange) {
                onBoundsChange(bounds);
            }
        }
    };

    private handleOutSideMobileCardClick = (): void => {
        const {onResetSelectedCard} = this.props;

        if (onResetSelectedCard) {
            onResetSelectedCard();
        }
    };

    private handleMapMarkerClick = (permalink: PermalinkType): void => {
        const {
            offerRequestParams,
            hotelWithOffersByPermalink,
            onClickMapMarker,
            isMobile,
            withTopHotel,
        } = this.props;
        const hotelWithOffers = hotelWithOffersByPermalink?.[permalink];

        if (hotelWithOffers) {
            if (!isMobile) {
                const searchParams = getSearchParamsForBaseHotelUrl({
                    seed: EPortalSeedReferer.PORTAL_HOTELS_SEARCH,
                    hotel: hotelWithOffers.hotel,
                    offers: hotelWithOffers.offers,
                    offerRequestParams,
                });
                const hotelUrl = hotelsURLs.getHotelUrl(searchParams);

                window.open(hotelUrl, '_blank');
            }

            if (onClickMapMarker) {
                onClickMapMarker(permalink);
            }
        }

        if (withTopHotel) {
            reachGoal(
                hotelWithOffers?.searchedByUser
                    ? EHotelsGoal.MAP_YELLOW_PIN_CLICK
                    : EHotelsGoal.MAP_OTHER_PIN_CLICK,
            );
        }
    };

    private handleMapClick = (): void => {
        const {
            experiments: {hotelsNewMapSnippet},
            isMobile,
        } = this.props;

        if (!hotelsNewMapSnippet || !isMobile) {
            return;
        }

        this.handleOutSideMobileCardClick();
    };

    private getHotelsToRender(): IHotelWithOffers[] {
        const {hotels, hotelWithOffersByPermalink} = this.props;

        return hotels
            .map((permalink: PermalinkType) => {
                const hotelWithOffers = hotelWithOffersByPermalink?.[permalink];

                if (
                    !hotelWithOffers ||
                    !this.canRenderMarkerByMapBounds(hotelWithOffers)
                ) {
                    return null;
                }

                return hotelWithOffers;
            })
            .filter(Boolean) as IHotelWithOffers[];
    }

    /* Render */

    private renderHotelsMarkers(
        hotelsWithOffers: IHotelWithOffers[],
    ): React.ReactNode {
        const {mapSuccessLoaded} = this.state;
        const {
            activeHotelPermalink,
            selectedMapMarkerPermalink,
            viewedMapMarkerPermalinks,
        } = this.props;

        if (!mapSuccessLoaded) {
            return null;
        }

        return hotelsWithOffers.map((hotelWithOffers: IHotelWithOffers) => (
            <HotelsMapMarker
                key={hotelWithOffers.hotel.permalink}
                hotelWithOffers={hotelWithOffers}
                isActiveMarker={
                    !selectedMapMarkerPermalink &&
                    activeHotelPermalink === hotelWithOffers.hotel.permalink
                }
                isSelectedMarker={
                    selectedMapMarkerPermalink ===
                    hotelWithOffers.hotel.permalink
                }
                isClickedMarker={
                    viewedMapMarkerPermalinks &&
                    viewedMapMarkerPermalinks[hotelWithOffers.hotel.permalink]
                }
                yandexMapAPIInstance={this.yandexMapAPIInstance}
                onSetMarkerPosition={this.handleChangeActiveMarkerPosition}
                onClickMapMarker={this.handleMapMarkerClick}
                onHoverMapMarker={this.handleHoverMapMarker}
            />
        ));
    }

    private renderToggleSearchTypeByMapBounds(): React.ReactNode {
        const {
            experiments: {hotelsHeaderRedesign},
            isMobile,
        } = this.props;
        const {canRenderSearchButton, canStartSearchAfterBoundsChange} =
            this.state;

        if (canRenderSearchButton) {
            return null;
        }

        if (isMobile && hotelsHeaderRedesign) {
            return null;
        }

        return (
            <div
                className={cx('toggleSearchByMapBounds')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'toggleSearchByMapBounds',
                })}
            >
                <Checkbox
                    label={i18nBlock.searchByMapBounds()}
                    size="s"
                    checked={canStartSearchAfterBoundsChange}
                    lines="one"
                    onChange={this.handleChangeSearchTypeByMapBounds}
                />
            </div>
        );
    }

    private renderSearchButton(): React.ReactNode {
        const {canRenderSearchButton} = this.state;

        if (!canRenderSearchButton) {
            return null;
        }

        return (
            <Button
                className={cx('searchButton')}
                onClick={this.handleSearchButtonClick}
                theme="raised"
            >
                {i18nBlock.searchButton()}
            </Button>
        );
    }

    private renderSelectedHotelCard(): React.ReactNode {
        const {
            isMobile,
            nights,
            offerRequestParams,
            hotelWithOffersByPermalink,
            selectedMapMarkerPermalink,
            onFavoriteClick,
        } = this.props;

        if (!isMobile) {
            return null;
        }

        const hotelWithOffers =
            selectedMapMarkerPermalink && hotelWithOffersByPermalink
                ? hotelWithOffersByPermalink[selectedMapMarkerPermalink]
                : undefined;

        return (
            <MobileMapHotelCard
                className={cx('mobileMapHotelCard')}
                offerRequestParams={offerRequestParams}
                hotelWithOffers={hotelWithOffers}
                nights={nights}
                onFavoriteClick={onFavoriteClick}
                onOutSideCardClick={this.handleOutSideMobileCardClick}
            />
        );
    }

    private renderHoveredHotelCard(): React.ReactNode {
        const {
            isMobile,
            hotelWithOffersByPermalink,
            activeHotelPermalink,
            activeHotelSource,
            experiments: {hotelsPercentDiscount},
        } = this.props;
        const {activeMarkerPosition, containerSize} = this.state;

        if (
            isMobile ||
            !activeHotelPermalink ||
            !activeMarkerPosition ||
            activeHotelSource !== EActiveHotelSource.MAP
        ) {
            return null;
        }

        const hotelWithOffers =
            hotelWithOffersByPermalink?.[activeHotelPermalink];

        if (!hotelWithOffers) {
            return null;
        }

        return (
            <MapHotelCard
                className={cx('hotelCard')}
                hotel={hotelWithOffers}
                containerSize={containerSize}
                markerPosition={activeMarkerPosition}
                hotelsPercentDiscount={hotelsPercentDiscount}
            />
        );
    }

    renderNoResultsMessage(
        hotelsWithOffers: IHotelWithOffers[],
    ): React.ReactNode {
        const {
            isSearchFinished,
            isMobile,
            totalActiveFilters,
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        if (!isMobile || !hotelsHeaderRedesign) {
            return null;
        }

        if (!isSearchFinished || hotelsWithOffers.length) {
            return null;
        }

        return (
            <NoResultsMessage
                noResults={!hotelsWithOffers.length}
                totalActiveFilters={totalActiveFilters}
            />
        );
    }

    renderSpinner(hotelsWithOffers: IHotelWithOffers[]): React.ReactNode {
        const {
            isLoading,
            isSearchFinished,
            isMobile,
            experiments: {hotelsHeaderRedesign},
        } = this.props;

        if (!isMobile || !hotelsHeaderRedesign) {
            return null;
        }

        // если после первого запроса пришли отели, но поллинг еще не закончился, то спиннер не показываем
        // иначе показываем спиннер пока не закончится поллинг или не появятся отели
        if (!isLoading && (isSearchFinished || hotelsWithOffers.length)) {
            return null;
        }

        return (
            <div className={cx('spinner')}>
                <Spinner size="s" />
            </div>
        );
    }

    render(): React.ReactNode {
        const {
            bboxAsStruct,
            className,
            mapClassName,
            mapHeight: height,
            mapWidth: width,
        } = this.props;

        if (!bboxAsStruct) {
            return null;
        }

        const hotelsWithOffers = this.getHotelsToRender();

        return (
            <div className={cx('hotelsMap', className)}>
                <YandexMaps
                    className={mapClassName}
                    defaultBounds={bboxAsStruct}
                    avoidFractionalZoom={false}
                    yandexMapDisablePoiInteractivity={true}
                    onActionBegin={this.handleMapActionBegin}
                    onActionEnd={this.handleMapActionEnd}
                    onStartChangeZoom={this.handleStartChangeZoom}
                    onBoundsChange={this.handleBoundsChange}
                    onClick={this.handleMapClick}
                    onClickDownZoomButton={this.handleChangeZoomByControlButton}
                    onClickUpZoomButton={this.handleChangeZoomByControlButton}
                    yandexMapInstanceRef={this.setYandexMapInstanceRef}
                    controls={MAP_CONTROLS}
                    onLoadMap={this.handleLoadMap}
                    hasGeoLocationControl
                    mapStyle={{height, width}}
                    {...prepareQaAttributes(this.props)}
                >
                    {this.renderHotelsMarkers(hotelsWithOffers)}
                    {this.renderSearchButton()}
                    {this.renderToggleSearchTypeByMapBounds()}
                </YandexMaps>

                {this.renderHoveredHotelCard()}
                {this.renderSelectedHotelCard()}
                {this.renderNoResultsMessage(hotelsWithOffers)}
                {this.renderSpinner(hotelsWithOffers)}
            </div>
        );
    }
}

export default connect(searchMapSelector)(HotelsSearchMap);
