import {PureComponent} from 'react';
import {renderToString} from 'react-dom/server';
import {IYMapsApi} from 'react-yandex-maps';

/* Types */
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';
import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {IHotelOffer} from 'types/hotels/offer/IHotelOffer';
import {IWithClassName} from 'types/withClassName';

import {formatPrice} from 'utilities/currency';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import {YandexMapPlacemark} from 'components/YandexMaps/YandexMaps';
import SellIcon from 'icons/24/Sell';

/* Init styles */
import cx from './HotelsMapMarker.scss';

interface IHotelsMapMarkerProps extends IWithClassName {
    hotelWithOffers: IHotelWithOffers;
    isActiveMarker?: boolean;
    isSelectedMarker?: boolean;
    isClickedMarker?: boolean;
    yandexMapAPIInstance: IYMapsApi | null;
    onHoverMapMarker?: (permalink: PermalinkType | undefined) => void;
    onClickMapMarker?: (permalink: PermalinkType) => void;
    onSetMarkerPosition?: (position: {
        top: number;
        left: number;
        bottom: number;
    }) => void;
}

const HOVER_Z_INDEX = 10;
const HAS_PRICE_Z_INDEX = 9;
const HAS_PROGRESS_Z_INDEX = 8;
const WITHOUT_PRICE_Z_INDEX = 1;

class HotelsMapMarker extends PureComponent<IHotelsMapMarkerProps> {
    /* Helpers */

    private getFirstHotelOffer = (
        offers: IHotelOffer[] | undefined,
    ): IHotelOffer | undefined => {
        if (offers && offers.length > 0) {
            const [firstOffer] = offers;

            return firstOffer;
        }
    };

    private getMarkerZIndex = (hotelWithOffers: IHotelWithOffers): number => {
        const {isActiveMarker} = this.props;
        const {offers, searchIsFinished} = hotelWithOffers;
        const firstOffer = this.getFirstHotelOffer(offers);

        if (isActiveMarker) {
            return HOVER_Z_INDEX;
        }

        if (firstOffer?.price) {
            return HAS_PRICE_Z_INDEX;
        }

        if (searchIsFinished) {
            return WITHOUT_PRICE_Z_INDEX;
        }

        return HAS_PROGRESS_Z_INDEX;
    };

    private resetMarkerHover = () => {
        const {onHoverMapMarker} = this.props;

        if (onHoverMapMarker) {
            onHoverMapMarker(undefined);
        }
    };

    /* Handlers */

    private handleMarkerMouseLeave = () => {
        this.resetMarkerHover();
    };

    private handleMarkerMouseEnter = () => {
        const {
            onHoverMapMarker,
            hotelWithOffers: {
                hotel: {permalink},
            },
        } = this.props;

        if (onHoverMapMarker) {
            onHoverMapMarker(permalink);
        }
    };

    private handleMarkerClick = () => {
        const {
            onClickMapMarker,
            hotelWithOffers: {
                hotel: {permalink},
            },
        } = this.props;

        if (onClickMapMarker) {
            onClickMapMarker(permalink);
        }

        this.resetMarkerHover();
    };

    private handleMarkerMouseDown = () => {
        this.resetMarkerHover();
    };

    private getMarkerStyleMods(baseClassName: string): string {
        const {isClickedMarker, isActiveMarker, isSelectedMarker} = this.props;

        return cx(baseClassName, {
            [`${baseClassName}_active`]: isActiveMarker || isSelectedMarker,
            [`${baseClassName}_visited`]: isClickedMarker,
        });
    }

    /* Render */

    private renderHotelMarker = (hotelWithOffers: IHotelWithOffers) => {
        const {
            offers,
            searchIsFinished,
            hotel: {permalink},
        } = hotelWithOffers;
        const firstOffer = this.getFirstHotelOffer(offers);
        const qaAttrs = prepareQaAttributes(
            `${permalink}-hotels-searchPage-map-marker`,
        );

        if (firstOffer?.price) {
            return (
                <div
                    className={this.getMarkerStyleMods('markerWithPrice')}
                    {...qaAttrs}
                >
                    {formatPrice({
                        ...firstOffer.price,
                        isCurrencyShown: true,
                    })}
                    {Boolean(firstOffer.discountInfo) && (
                        <SellIcon
                            width="16"
                            height="16"
                            className={cx('markerIcon')}
                        />
                    )}
                </div>
            );
        }

        if (searchIsFinished) {
            return (
                <div
                    className={this.getMarkerStyleMods('markerWithoutPrice')}
                    {...qaAttrs}
                />
            );
        }

        return (
            <div
                className={this.getMarkerStyleMods('markerHasProgress')}
                {...qaAttrs}
            />
        );
    };

    render() {
        const {
            hotelWithOffers,
            isActiveMarker,
            yandexMapAPIInstance,
            onSetMarkerPosition,
        } = this.props;
        const {
            hotel: {permalink, coordinates},
        } = hotelWithOffers;
        const hotelMarkerNode = this.renderHotelMarker(hotelWithOffers);

        if (!yandexMapAPIInstance || !hotelMarkerNode) return null;

        const markerContentLayout =
            yandexMapAPIInstance.templateLayoutFactory.createClass(
                renderToString(hotelMarkerNode),
                {
                    build: function () {
                        this.constructor.superclass.build.call(this);
                        this._markerNode = this.getElement().firstChild;
                        this._parentElementNode = this.getParentElement();
                    },
                    getShape: function () {
                        const {
                            offsetWidth: markerWidth,
                            offsetHeight: markerHeight,
                        } = this._markerNode;

                        // Добавил проверку на markerHeight чтобы избежать лишних апдейтов
                        // когда убираем маркер с карточки
                        if (
                            isActiveMarker &&
                            onSetMarkerPosition &&
                            markerHeight
                        ) {
                            const {offsetTop, offsetLeft} =
                                this._parentElementNode;

                            onSetMarkerPosition({
                                top: offsetTop - markerHeight,
                                left: offsetLeft,
                                bottom: offsetTop,
                            });
                        }

                        return new yandexMapAPIInstance.shape.Rectangle(
                            new yandexMapAPIInstance.geometry.pixel.Rectangle([
                                [markerWidth / 2, -markerHeight],
                                [-markerWidth / 2, 0],
                            ]),
                        );
                    },
                },
            );

        return (
            <YandexMapPlacemark
                key={permalink}
                geometry={[coordinates.lat, coordinates.lon]}
                onClick={this.handleMarkerClick}
                onMouseEnter={this.handleMarkerMouseEnter}
                onMouseLeave={this.handleMarkerMouseLeave}
                onMouseDown={this.handleMarkerMouseDown}
                options={{
                    iconLayout: markerContentLayout,
                    zIndex: this.getMarkerZIndex(hotelWithOffers),
                }}
            />
        );
    }
}

export default HotelsMapMarker;
