import React, {useCallback, useMemo} from 'react';
import {useHistory} from 'react-router-dom';
import ReactTruncate from 'react-truncate';

import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';
import {
    IHotelOffer,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {EPortalSeedReferer} from 'types/hotels/common/IAttribution';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {convertHotelOrRoomImages} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {
    getSearchParamsForBaseHotelUrl,
    hotelsURLs,
} from 'projects/depreacted/hotels/utilities/urls';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotels-SearchHotelCard';

import Card from 'components/Card/Card';
import Text from 'components/Text/Text';
import HotelRatingAndCategory from 'projects/depreacted/hotels/components/HotelRatingAndCategory/HotelRatingAndCategory';
import Price from 'components/Price/Price';
import PlusPoints from 'components/PlusPoints/PlusPoints';
import StriketroughText from 'components/StriketroughText/StriketroughText';
import Favorites from 'projects/depreacted/hotels/components/Favorites/Favorites';
import ImagesSwipeCarousel from 'components/ImagesSwipeCarousel/ImagesSwipeCarousel';
import HotelName from 'projects/depreacted/hotels/components/HotelName/HotelName';
import Link from 'components/Link/Link';
import HotelImageStub from 'projects/depreacted/hotels/components/HotelImageStub/HotelImageStub';

import cx from './MobileMapHotelSnippet.scss';

interface IFavoriteEvent {
    permalink: PermalinkType;
    isFavorite: boolean;
}

interface IMobileMapHotelSnippetProps {
    hotelWithOffers: IHotelWithOffers;
    offerRequestParams?: IRequiredOfferParams;

    nights: number;

    isFavorite: boolean;
    onFavoriteClick?: (event: IFavoriteEvent) => void;
}

const IMAGES_SIZE_NAME = 'M';
const STUB_PROPS = {width: 64, height: 64};

const MobileMapHotelSnippet: React.FC<IMobileMapHotelSnippetProps> = ({
    nights,
    isFavorite,
    onFavoriteClick,
    hotelWithOffers,
    offerRequestParams,
}) => {
    const history = useHistory();
    const deviceType = useDeviceType();
    const {hotel, offers} = hotelWithOffers;
    const {permalink, name, images} = hotel;

    const convertedImages = useMemo(
        () => convertHotelOrRoomImages(images, IMAGES_SIZE_NAME),
        [images],
    );

    const searchParams = getSearchParamsForBaseHotelUrl({
        seed: EPortalSeedReferer.PORTAL_HOTELS_SEARCH,
        hotel: hotelWithOffers.hotel,
        offers: hotelWithOffers.offers,
        offerRequestParams,
    });
    const hotelUrl = hotelsURLs.getHotelUrl(searchParams);

    const [firstOffer]: IHotelOffer[] = offers || [];

    const handleFavoritesClick = useCallback(() => {
        if (!onFavoriteClick) {
            return;
        }

        onFavoriteClick({permalink, isFavorite});

        reachGoal(EFavoritesGoal.FAVORITES_CLICK);
        reachGoal(EHotelsGoal.SEARCHMAP_CARD_FAV);
    }, [isFavorite, onFavoriteClick, permalink]);

    const handleCardClick = useCallback(() => {
        // Общая метрика для выдачи + карты
        reachGoal(EHotelsGoal.SEARCH_PAGE_CARD_HOTEL_CLICK);
        reachGoal(EHotelsGoal.SEARCHMAP_CARD_CLICK);
    }, []);

    const handleImageClick = useCallback(() => {
        handleCardClick();
        history.push(hotelUrl);
    }, [hotelUrl, history, handleCardClick]);

    const handleImageSwipe = useCallback(() => {
        // Общая метрика для выдачи + карты
        reachGoal(EHotelsGoal.SEARCH_PAGE_IMAGE_SWIPE);

        reachGoal(EHotelsGoal.SEARCHMAP_CARD_IMAGE_SWIPE);
    }, []);

    const carousel = useMemo(() => {
        if (images.length === 0) {
            return (
                <HotelImageStub
                    onClick={handleImageClick}
                    className={cx('stub')}
                    illustrationProps={STUB_PROPS}
                />
            );
        }

        return (
            <ImagesSwipeCarousel
                maxItemsCount={3}
                key={hotel.hotelSlug}
                borderRadius="s"
                swipeableClassName={cx('carousel')}
                className={cx('carouselRoot')}
                onImageClick={handleImageClick}
                images={convertedImages}
                onScrollContent={handleImageSwipe}
            />
        );
    }, [images, hotel, handleImageClick, convertedImages, handleImageSwipe]);

    return (
        <Card shadow="popup" radius="m" className={cx('root')}>
            {carousel}
            <div className={cx('favorites')}>
                <Favorites
                    deviceType={deviceType}
                    isFavorite={isFavorite}
                    onClick={handleFavoritesClick}
                />
            </div>
            <div className={cx('offerContainer')}>
                <Link
                    to={hotelUrl}
                    className={cx('link')}
                    rel="noopener noreferrer"
                    onClick={handleCardClick}
                />
                <Text tag="div" weight="bold" className={cx('title')}>
                    <ReactTruncate lines={2}>
                        <HotelName hotelName={name} />
                    </ReactTruncate>
                </Text>
                <HotelRatingAndCategory
                    hotel={hotel}
                    size="s"
                    hotelReviewsUrl={hotelUrl}
                />
                <div className={cx('offer')}>
                    <Price
                        color={firstOffer.discountInfo ? 'alert' : undefined}
                        className={cx('price')}
                        value={firstOffer.price.value}
                        currency={firstOffer.price.currency}
                    />
                    {firstOffer.discountInfo && (
                        <StriketroughText
                            size="m"
                            crossed
                            theme="alert"
                            className={cx('discountedPrice')}
                        >
                            <Price
                                color="secondary"
                                value={
                                    firstOffer.discountInfo.strikethroughPrice
                                        .value
                                }
                                currency={
                                    firstOffer.discountInfo.strikethroughPrice
                                        .currency
                                }
                            />
                        </StriketroughText>
                    )}
                    {firstOffer.offerYandexPlusInfo && (
                        <PlusPoints
                            size="s"
                            weight="medium"
                            count={firstOffer.offerYandexPlusInfo.points}
                            className={cx('plusPoints')}
                        />
                    )}
                </div>
                <Text size="s" tag="div">
                    {i18nBlock.nightsCount({
                        nightsCount: nights,
                    })}
                </Text>
            </div>
        </Card>
    );
};

export default MobileMapHotelSnippet;
