import {FC, ReactElement, useCallback} from 'react';

/* Components */
import {SearchPageLayoutView} from 'types/hotels/search/ISearchLayout';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/hotels-SearchPage';

import Button from 'components/Button/Button';
import MapIcon from 'icons/16/Map';
import ListIcon from 'icons/16/List';

import cx from './MapListRadioButton.scss';

/* Init styles */

/* Components Types */
interface IMapListRadioButton extends IWithClassName, IWithQaAttributes {
    activeView: SearchPageLayoutView;
    onChangeActiveView: (newValue: SearchPageLayoutView) => void;
}

const SEARCH_PAGE_VIEWS_NAMES: Record<SearchPageLayoutView, string> = {
    [SearchPageLayoutView.LIST]: i18nBlock.hotelsTab(),
    [SearchPageLayoutView.MAP]: i18nBlock.mapTab(),
};

const ICON_BY_TYPE: Record<SearchPageLayoutView, ReactElement> = {
    [SearchPageLayoutView.LIST]: <ListIcon />,
    [SearchPageLayoutView.MAP]: <MapIcon />,
};

const getOppositeValue = (v: SearchPageLayoutView): SearchPageLayoutView =>
    v === SearchPageLayoutView.LIST
        ? SearchPageLayoutView.MAP
        : SearchPageLayoutView.LIST;

const MapListRadioButton: FC<IMapListRadioButton> = props => {
    const {className, activeView, onChangeActiveView} = props;

    const oppositeValue = getOppositeValue(activeView);
    const deviceType = useDeviceType();

    const handleClick = useCallback(() => {
        onChangeActiveView(oppositeValue);
    }, [oppositeValue, onChangeActiveView]);

    return (
        <Button
            className={cx(
                'button',
                deviceModMobile('button', deviceType),
                className,
            )}
            value={activeView}
            size="l"
            onClick={handleClick}
            theme="outlined"
            iconLeft={ICON_BY_TYPE[oppositeValue]}
            {...prepareQaAttributes({
                parent: props,
            })}
        >
            {SEARCH_PAGE_VIEWS_NAMES[oppositeValue]}
        </Button>
    );
};

export default MapListRadioButton;
