import {
    ISearchPageQueryFilterParams,
    ISearchPageQueryMapParams,
    ISearchPageGeoQueryParams,
    ISearchPageQueryParams,
    ISearchPageNavigationQueryParams,
    ISearchPageSortQueryParams,
    ISearchPageGeolocationQueryParams,
    ISearchPageTopHotelQueryParams,
    ISearchPagePollingParams,
} from 'types/hotels/common/IQueryParams';
import {MapBoundsType} from 'types/common/ICoordinates';
import {GeoIdType} from 'types/hotels/hotel/IGeoRegion';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {IFullFilters} from 'types/hotels/search/IFiltersInfo';
import {NavigationTokenType} from 'types/hotels/search/INavigationTokens';
import {TSortId} from 'types/hotels/search/ISortInfo';
import {HotelSlugType} from 'types/hotels/hotel/IHotel';

import {convertBoundsToString} from 'components/YandexMaps/utilities';
import {getOfferSearchQueryParams} from 'projects/depreacted/hotels/utilities/getRequiredOfferSearchQuery/getRequiredOfferSearchQuery';

export interface ISearchPageInfoForQuery {
    offerSearchParams?: IRequiredOfferParams;
    geoId?: GeoIdType;
    mapBounds?: MapBoundsType;
    filters: IFullFilters;
    navigationToken?: NavigationTokenType;
    selectedSortId?: TSortId;
    sortOrigin?: string;
    topHotelSlug?: HotelSlugType;
    searchPagePollingId?: string;
}

/* GeoRegion */
export const getGeoQueryParams = (
    geoId?: GeoIdType,
): ISearchPageGeoQueryParams | undefined => {
    if (geoId) {
        return {
            geoId,
        };
    }
};

export const getGeolocationQueryParams = (
    sortOrigin?: string,
): ISearchPageGeolocationQueryParams | undefined => {
    if (sortOrigin) {
        return {
            sortOrigin,
        };
    }
};

/* MapBounds */
export const getMapQueryParams = (
    mapBounds?: MapBoundsType,
): ISearchPageQueryMapParams | undefined => {
    if (mapBounds) {
        return {
            bbox: convertBoundsToString(mapBounds),
        };
    }
};

/* Filters */
export const getFilterQueryParams = (
    permanentFilters: IFullFilters,
): Partial<ISearchPageQueryFilterParams> => {
    const {activeFilterAtoms, priceFilter} = permanentFilters;

    const filterQueryParams: Partial<ISearchPageQueryFilterParams> = {};

    if (activeFilterAtoms) {
        const activeFilterAtomsList = Object.keys(activeFilterAtoms);

        if (activeFilterAtomsList.length) {
            filterQueryParams.filterAtoms = activeFilterAtomsList.join(',');
        }
    }

    if (priceFilter) {
        const {minValue, maxValue, minPriceEstimate, maxPriceEstimate} =
            priceFilter;

        if (minValue && minValue !== minPriceEstimate) {
            filterQueryParams.minPrice = minValue;
        }

        if (maxValue && maxValue !== maxPriceEstimate) {
            filterQueryParams.maxPrice = maxValue;
        }
    }

    return filterQueryParams;
};

/* NavigationToken */
export const getNavigationQueryParams = (
    navigationToken?: NavigationTokenType,
): ISearchPageNavigationQueryParams | undefined => {
    if (navigationToken) {
        return {
            navigationToken,
        };
    }
};

/* Sort */
export const getSortQueryParams = (
    selectedSortId?: TSortId,
): ISearchPageSortQueryParams | undefined => {
    if (selectedSortId) {
        return {
            selectedSortId,
        };
    }

    return undefined;
};

/* Top hotel */
const getTopHotelQueryParams = (
    topHotelSlug?: HotelSlugType,
): ISearchPageTopHotelQueryParams | undefined => {
    return topHotelSlug ? {topHotelSlug} : undefined;
};

/* PollingId */
const getPollingQueryParams = (
    searchPagePollingId?: string,
): ISearchPagePollingParams | undefined => {
    return searchPagePollingId ? {searchPagePollingId} : undefined;
};

/* Main */
export const getSearchPageQueryByInfo = ({
    geoId,
    offerSearchParams,
    mapBounds,
    filters,
    navigationToken,
    selectedSortId,
    sortOrigin,
    topHotelSlug,
    searchPagePollingId,
}: ISearchPageInfoForQuery): ISearchPageQueryParams => {
    const filterQuery = getFilterQueryParams(filters);
    const mapQuery = getMapQueryParams(mapBounds);
    const offerSearchQuery = getOfferSearchQueryParams(offerSearchParams);
    const geoSearchQuery = getGeoQueryParams(geoId);
    const geolocationSearchQuery = getGeolocationQueryParams(sortOrigin);
    const navigationTokenQuery = getNavigationQueryParams(navigationToken);
    const sortQuery = getSortQueryParams(selectedSortId);
    const topHotelQuery = getTopHotelQueryParams(topHotelSlug);
    const searchPagePollingIdQuery = getPollingQueryParams(searchPagePollingId);

    return {
        ...filterQuery,
        ...mapQuery,
        ...offerSearchQuery,
        ...geoSearchQuery,
        ...geolocationSearchQuery,
        ...navigationTokenQuery,
        ...sortQuery,
        ...topHotelQuery,
        ...searchPagePollingIdQuery,
    };
};
