import {
    IHotelOrRoomImage,
    IHotelOrRoomImageSize,
} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageCustomSize} from 'projects/depreacted/hotels/pages/HotelPage/components/Gallery/types';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';
import {EHotelImageYapicSize} from 'types/hotels/common/EHotelImageYapicSize';
import {TImage, TResponsiveImage} from 'types/common/TImage';

/* Types */
interface IImagesUrlTemplateType {
    urlTemplate: string;
    sizes: IHotelOrRoomImageSize[];
}

/* Utilities */
export const pickSuitableSize = (
    imageSizeName: string,
    availableSizes: IHotelOrRoomImageSize[] | null,
): string => {
    if (!availableSizes) return imageSizeName;

    const desiredSize = availableSizes.find(
        info => info.size === imageSizeName,
    )?.size;
    const fallbackSize = availableSizes.find(info => info.size)?.size;

    const defaultSize = imageSizeName;

    return desiredSize || fallbackSize || defaultSize;
};

export const parseImageUrlTemplate = (
    imageUrlTemplate: string,
    imageSizeName: string = EHotelImageCustomSize.L,
    availableSizes: IHotelOrRoomImageSize[] | null,
    replacePattern = /%s$/,
): string => {
    return imageUrlTemplate.replace(
        replacePattern,
        pickSuitableSize(imageSizeName, availableSizes),
    );
};

export const parseAltaiImageUrlTemplate = (
    imageUrlTemplate: string,
    imageSizeName: EHotelImageAltayStandardSize | string,
    availableSizes: IHotelOrRoomImageSize[],
): string =>
    parseImageUrlTemplate(
        imageUrlTemplate,
        imageSizeName,
        availableSizes,
        /{size}$/,
    );

export const parseYapicImageUrlTemplate = (
    imageUrlTemplate: string,
    imageSizeName: EHotelImageYapicSize,
): string =>
    parseImageUrlTemplate(imageUrlTemplate, imageSizeName, null, /{size}$/);

export const convertImagesToFlatList = (
    images: TImage[],
    imageSizeName: string,
    availableSizes: IHotelOrRoomImageSize[],
): string[] =>
    (images || []).map(image =>
        parseImageUrlTemplate(image.src, imageSizeName, availableSizes),
    );

export const convertUrlTemplateImagesToFlatList = (
    images: IImagesUrlTemplateType[],
    imageSizeName: string,
    imageSrcGetter?: (
        urlTemplate: string,
        size: any,
        availableSizes: IHotelOrRoomImageSize[],
    ) => string,
): string[] => {
    const imageGetter =
        typeof imageSrcGetter === 'function'
            ? imageSrcGetter
            : parseImageUrlTemplate;

    return (images || []).map(image =>
        imageGetter(image.urlTemplate, imageSizeName, image.sizes),
    );
};

function mapAdditionalSizes(
    image: IHotelOrRoomImage,
    minImageSize?: IHotelOrRoomImageSize,
): TImage[] {
    return image.sizes
        .filter(size => !minImageSize || size.width >= minImageSize.width)
        .map(size => ({
            src: parseImageUrlTemplate(image.urlTemplate, size.size, [size]),
            size,
        }));
}

/** Конвертация модели изображений для использования в карусели */
export function convertHotelOrRoomImages(
    images: IHotelOrRoomImage[],
    imageSizeName: string,
): TResponsiveImage[] {
    return images.map(image => {
        const sizeName = pickSuitableSize(imageSizeName, image.sizes);
        const size = image.sizes.find(info => info.size === sizeName);

        return {
            src: parseImageUrlTemplate(
                image.urlTemplate,
                imageSizeName,
                image.sizes,
            ),
            size: size,
            additionalSizes: mapAdditionalSizes(image, size),
        };
    });
}
