import {Location} from 'history';
import _isUndefined from 'lodash/isUndefined';

import {
    AWAITS_PAYMENT,
    CANCELLED,
    CANCELLED_WITH_REFUND,
    CONFIRMED,
    FAILED,
    NEED_AUTHORIZATION,
    PAYMENT_FAILED,
    REFUND_FAILED,
    REFUNDED,
    UNKNOWN_ORDER_ERROR,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';
import {URLs} from 'constants/urls';

import {getInvalidInputDetailsFieldType} from 'selectors/depreacted/hotels/book/orderCancellationDetails/getInvalidInputDetailsFieldType';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {internalUrl} from 'utilities/url';

interface IOfferPageQuery {
    hotelPermalink: number;
    startDate: string;
    endDate: string;
    adults: number;
    childrenAges: number;
    searchType: string;
}

interface ITokenPageQuery {
    token?: string | null;
    label?: string | null;
}

interface IOrderPageQuery {
    orderId?: string | null;
}

type TPageQuery =
    | IOfferPageQuery
    | IOrderPageQuery
    | ITokenPageQuery
    | undefined;

interface IGetPathByOrderStatusProps {
    orderId: string | undefined;
    orderStatus: string;
    orderCancellationDetails?: any;
    location: Location;
}

// TODO: в функции есть скрытая логика, что при совпадении текущего url с тем,
// который генерирует функция, то она возвращает undefined
export function getPathByOrderStatus(
    props: IGetPathByOrderStatusProps,
): string | undefined {
    const {orderId, orderStatus, orderCancellationDetails, location} = props;
    let nextPagePath = '';
    let pageBaseQuery: TPageQuery = {orderId};

    switch (orderStatus) {
        case AWAITS_PAYMENT: {
            nextPagePath = URLs.hotelsBookPayment;

            break;
        }

        case PAYMENT_FAILED: {
            nextPagePath = URLs.hotelsBookPayment;

            break;
        }

        case CONFIRMED: {
            nextPagePath = URLs.hotelsBookSuccess;

            break;
        }

        case UNKNOWN_ORDER_ERROR: {
            nextPagePath = URLs.hotelsBookError;

            if (_isUndefined(orderId)) {
                pageBaseQuery = getTokenAndLabelByLocation(location);
            }

            break;
        }

        case REFUNDED: {
            nextPagePath = URLs.hotelsCancelledOrder;

            break;
        }

        case REFUND_FAILED: {
            nextPagePath = URLs.hotelsCancelledOrder;

            break;
        }

        case NEED_AUTHORIZATION: {
            nextPagePath = URLs.hotelsBookError;
            pageBaseQuery = getOrderIdByLocation(location);

            break;
        }

        case CANCELLED: {
            if (isEmailInvalidCancelledStatus(orderCancellationDetails)) {
                nextPagePath = URLs.hotelsBook;

                break;
            }

            nextPagePath = URLs.hotelsBookError;

            break;
        }
        case CANCELLED_WITH_REFUND:
        case FAILED: {
            nextPagePath = URLs.hotelsBookError;

            break;
        }
    }

    return getHotelsPath(location, nextPagePath, pageBaseQuery);
}

function isEmailInvalidCancelledStatus(
    orderCancellationDetails: any | undefined,
): boolean {
    const {reason} = orderCancellationDetails || {};

    return (
        reason === 'OrderCancellationDetailsReasonType.INVALID_INPUT' &&
        getInvalidInputDetailsFieldType(orderCancellationDetails) ===
            'OrderCancellationDetailsFieldType.EMAIL'
    );
}

function getOrderIdByLocation(location: Location): IOrderPageQuery {
    const {orderId} = getQueryByLocation(location) as IOrderPageQuery;

    return {
        orderId,
    };
}

function getTokenAndLabelByLocation(location: Location): ITokenPageQuery {
    const {token, label} = getQueryByLocation(location) as ITokenPageQuery;

    return {
        token,
        label,
    };
}

function getHotelsPath(
    location: Location,
    nextPagePath: string,
    pageBaseQuery: TPageQuery,
): string | undefined {
    const {pathname} = location;

    if (isSamePathname(pathname, nextPagePath)) {
        return internalUrl(nextPagePath, pageBaseQuery);
    }

    return undefined;
}

function isSamePathname(prevPathname: string, nextPagePath: string): boolean {
    return Boolean(nextPagePath && internalUrl(nextPagePath) !== prevPathname);
}
