import {ParsedQuery} from 'query-string';

import {DEFAULT_CHILDREN_AGES, DEFAULT_ADULTS_COUNT} from 'constants/hotels';

import {DateType} from 'types/hotels/common/IDate';
import {GeoIdType} from 'types/hotels/hotel/IGeoRegion';

type SearchParamQueryType = string[] | string | null | undefined;

/* Types */
export interface IValidateSearchParams {
    adults: number;
    geoId?: GeoIdType;
    startDate?: DateType;
    endDate?: DateType;
    checkinDate?: DateType;
    checkoutDate?: DateType;
    childrenAges: number[];
    hotelPermalink?: string;
    sortOrigin?: string;
    searchPagePollingId?: string;
}

export const parseChildrenAges = (
    childrenAges: SearchParamQueryType,
): number[] => {
    if (childrenAges && typeof childrenAges === 'string') {
        return childrenAges
            .split(',')
            .map(childAge => Number.parseInt(childAge, 10));
    }

    return DEFAULT_CHILDREN_AGES;
};

export const parseAdults = (adults: SearchParamQueryType): number => {
    if (adults && typeof adults === 'string') {
        return Number.parseInt(adults, 10);
    }

    return DEFAULT_ADULTS_COUNT;
};

export const parseGeoId = (
    geoId: SearchParamQueryType,
): GeoIdType | undefined => {
    if (geoId && typeof geoId === 'string') {
        return Number.parseInt(geoId, 10);
    }
};

export const validateSearchHotelsAdultsParam = (
    adults: number,
    checkinDate: string | undefined,
    checkoutDate: string | undefined,
): number | undefined => {
    const hasDates = checkinDate && checkoutDate;

    // бекенд четырехсотит если отправить ему adults и не отправлять checkinDate и checkoutDate
    return hasDates ? adults : undefined;
};

// todo add validate for startDate, endDate, adults, childrenAges!!!
export const validateSearchHotelsParams = (
    query: ParsedQuery,
): IValidateSearchParams => {
    const {
        adults,
        geoId,
        startDate,
        endDate,
        checkinDate,
        checkoutDate,
        childrenAges,
        hotelPermalink,
        sortOrigin,
        searchPagePollingId,
    } = query;

    return {
        geoId: parseGeoId(geoId),
        adults: parseAdults(adults),
        childrenAges: parseChildrenAges(childrenAges),
        // TODO: убрать startDate как будет убран или обновлен RedirectSearchController
        startDate: typeof startDate === 'string' ? startDate : undefined,
        // TODO: убрать endDate как будет убран или обновлен RedirectSearchController
        endDate: typeof endDate === 'string' ? endDate : undefined,
        checkinDate: typeof checkinDate === 'string' ? checkinDate : undefined,
        checkoutDate:
            typeof checkoutDate === 'string' ? checkoutDate : undefined,
        hotelPermalink:
            typeof hotelPermalink === 'string' ? hotelPermalink : undefined,
        sortOrigin: typeof sortOrigin === 'string' ? sortOrigin : undefined,
        searchPagePollingId:
            typeof searchPagePollingId === 'string'
                ? searchPagePollingId
                : undefined,
    };
};
