import React, {useEffect, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {ParsedQuery} from 'query-string';

import {EProjectName} from 'constants/common';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import searchResultsReducer from 'reducers/avia/search/reducer';
import hotelsPersonalizationReducer from 'reducers/depreacted/hotels/personalization/reducer';
import personalizationReducer from 'reducers/avia/personalization/reducer';
import calendarPricesReducer from 'reducers/avia/aviaCalendarPrices/reducer';
import aviaLogMetricsReducer from 'reducers/avia/aviaLogMetrics/reducer';
import {FeaturesState} from 'reducers/common/commonReducerTypes';
import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import nonceSelector from 'selectors/common/nonceSelector';
import crossLinksSelector from 'selectors/common/crossLinksSelector';

import indexAviaSaga from 'sagas/index/avia/rootSaga';
import pricesSaga from 'sagas/avia/aviaCalendarPrices/aviaCalendarPricesSaga';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import FeaturedDirectionsBlock from 'components/FeaturedDirectionsBlock/FeaturedDirectionsBlock';
import AviaAdvantages from 'projects/avia/components/AviaAdvantages/AviaAdvantages';
import AviaIndexTexts from 'projects/avia/components/AviaIndexTexts/AviaIndexTexts';
import IndexDirect from 'projects/index/IndexApp/components/IndexDirect/IndexDirect';
import SearchPresets from './components/SearchPresets/SearchPresets';

import {AviaRecipes} from 'projects/avia/containers/AviaRecipesContainer/AviaRecipes';
import SubscriptionContainer from 'projects/index/IndexApp/containers/SubscriptionContainer/SubscriptionContainer';

import cx from './IndexAvia.scss';

const ROOT_QA = 'indexAviaPage';

interface IIndexAviaOwnProps {
    query: ParsedQuery;
}

export interface IIndexAviaStatePropsContainer {
    features: FeaturesState;
}

async function prefetchData(): Promise<void> {
    await Promise.all([]);
}

const IndexAvia: React.FC<IIndexAviaOwnProps> = () => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const {upperSubscriptionForm} = useExperiments();
    const nonce = useSelector(nonceSelector);

    const {
        response: {crossLinks, recipes},
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = useSelector(crossLinksSelector);
    const needToRender = useServerDataFetcher([prefetchData]);

    const searchFormFieldsRef = useRef<IFormFieldsRef>();

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    useEffect(() => {
        if (project !== EProjectName.AVIA || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.AVIA));
        }
    }, [dispatch, isRecipesFetched, project]);

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.AVIA;

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            <SearchPresets
                className={cx('searchPresets', 'searchPresets_type-bottom')}
                searchFormFieldsRef={searchFormFieldsRef}
            />

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            {upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Avia}
                />
            )}

            <AviaAdvantages
                className={cx('advantages')}
                {...prepareQaAttributes(ROOT_QA)}
            />

            {renderCrossaleBlock && (
                <AviaRecipes
                    className={cx('recipes')}
                    recipes={recipes}
                    isLoading={isRecipesLoading}
                    deviceType={deviceType}
                    nonce={nonce}
                />
            )}

            <IndexDirect
                className={cx('direct')}
                projectName={EProjectName.AVIA}
            />

            <AviaIndexTexts
                className={cx('texts')}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'texts',
                })}
            />

            {renderCrossaleBlock && (
                <FeaturedDirectionsBlock
                    blockClassName={cx('featureBlock')}
                    projectName={project}
                    crossLinks={crossLinks}
                />
            )}

            {!upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Avia}
                />
            )}
        </div>
    );
};

export default withReducers<IIndexAviaOwnProps>([
    [ELoadableReducer.AVIA_SEARCH, searchResultsReducer],
    [ELoadableReducer.AVIA_CALENDAR_PRICES, calendarPricesReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
    [ELoadableReducer.AVIA_LOG_METRICS, aviaLogMetricsReducer],
    [ELoadableReducer.HOTELS_PERSONALIZATION, hotelsPersonalizationReducer],
    [ELoadableReducer.AVIA_PERSONALIZATION, personalizationReducer],
])(withSaga<IIndexAviaOwnProps>([indexAviaSaga, pricesSaga])(IndexAvia));
