import React, {MutableRefObject, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {resetAviaSearchHistoryAction} from 'reducers/avia/personalization/actions';

import {hotelsPersonalizationSelector} from 'selectors/depreacted/hotels/personalization/hotelsPersonalizationSelector';
import {aviaPersonalizationSearchHistory} from 'selectors/avia/personalization/aviaPersonalizationSearchHistory';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import formatSearchHistoryItemsToSearchFormItem from '../../../../utilities/formatSearchHistoryItemsToSearchFormItem';
import convertSearchHistoryToAviaMinPriceBatchSearchParams from 'projects/index/IndexApp/blocks/IndexAvia/components/SearchPresets/utilities/convertSearchHistoryToAviaMinPriceBatchSearchParams';
import findPrice from 'projects/index/IndexApp/blocks/IndexAvia/components/SearchPresets/utilities/findPrice';

import Flex from 'components/Flex/Flex';
import PreviousSearchPreset from './components/PreviousSearchPreset/PreviousSearchPreset';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import CrossSalePreset from './components/CrossSalePreset/CrossSalePreset';
import useMinPriceForSearchPresets from 'projects/index/IndexApp/blocks/IndexAvia/components/SearchPresets/hooks/useMinPriceForSearchPresets';

interface ISearchPresets extends IWithClassName, IWithQaAttributes {
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined>;
}

const SearchPresets: React.FC<ISearchPresets> = props => {
    const {className, searchFormFieldsRef} = props;

    const rootQa = getQa(props);
    const deviceType = useDeviceType();

    const dispatch = useDispatch();
    const {suggests} = useSelector(hotelsPersonalizationSelector);
    const {searchHistory} = useSelector(aviaPersonalizationSearchHistory);

    const {fetch, prices, isLoading} = useMinPriceForSearchPresets();

    useEffect(() => {
        const request =
            convertSearchHistoryToAviaMinPriceBatchSearchParams(searchHistory);

        if (!request.minRequests.length) {
            return;
        }

        fetch(request);

        return (): void => {
            dispatch(resetAviaSearchHistoryAction());
        };
    }, [dispatch, searchHistory, fetch]);

    const crossSaleItem = suggests.items.find(item => item);

    const itemsCount = (crossSaleItem ? 1 : 0) + searchHistory.items.length;

    const content = useMemo(() => {
        if (!itemsCount) {
            return null;
        }

        const width = deviceType.isMobile && itemsCount === 1 ? 'max' : 'auto';

        return (
            <Flex
                className={className}
                between={2}
                inline
                nowrap
                {...prepareQaAttributes(rootQa)}
            >
                {crossSaleItem && (
                    <CrossSalePreset item={crossSaleItem} width={width} />
                )}

                {searchHistory.items.map((item, index) => {
                    const price = findPrice(prices, item);

                    return (
                        <PreviousSearchPreset
                            key={index}
                            item={formatSearchHistoryItemsToSearchFormItem(
                                item,
                            )}
                            searchFormFieldsRef={searchFormFieldsRef}
                            width={width}
                            isLoading={isLoading}
                            price={price}
                        />
                    );
                })}
            </Flex>
        );
    }, [
        itemsCount,
        deviceType,
        className,
        rootQa,
        crossSaleItem,
        searchHistory.items,
        searchFormFieldsRef,
        prices,
        isLoading,
    ]);

    if (!content) {
        return null;
    }

    if (itemsCount === 1 || deviceType.isDesktop) {
        return content;
    }

    return <HorizontalScroller>{content}</HorizontalScroller>;
};

export default React.memo(SearchPresets);
