import React, {useEffect} from 'react';
import {useLocation} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {BUSES_HOW_TO_BUY_STEPS} from 'projects/index/IndexApp/blocks/IndexBuses/constants/howToBuySteps';

import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import contextReducer from 'reducers/buses/context/reducer';
import calendarReducer from 'reducers/buses/calendar/reducer';
import searchFormReducer from 'reducers/buses/searchForm/reducer';
import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import searchSuggestsReducer from 'reducers/buses/searchSuggests/reducer';
import previousSearchesReducer from 'reducers/buses/previousSearches/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import crossLinksSelector from 'selectors/common/crossLinksSelector';

import requestPreviousSearchesSaga from 'sagas/index/buses/previousSearches/requestPreviousSearchesSaga';

import {reachGoal} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import prefetchData from 'projects/index/IndexApp/blocks/IndexBuses/utilities/prefetchData';
import getIndexQuery from 'projects/index/IndexApp/blocks/IndexBuses/utilities/getIndexQuery';

import * as i18nBusesIndexBlock from 'i18nNew/buses-indexPage';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import Heading from 'components/Heading/Heading';
import HowToBuy from 'components/HowToBuy/HowToBuy';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import SearchForm from 'projects/buses/components/SearchForm/SearchForm';
import FeaturedDirectionsBlock from 'components/FeaturedDirectionsBlock/FeaturedDirectionsBlock';
import IndexDirect from 'projects/index/IndexApp/components/IndexDirect/IndexDirect';
import BusesAdvantages from 'projects/buses/components/BusesAdvantages/BusesAdvantages';
import PreviousSearches from 'projects/index/IndexApp/blocks/IndexBuses/components/PreviousSearches/PreviousSearches';

import SubscriptionContainer from 'projects/index/IndexApp/containers/SubscriptionContainer/SubscriptionContainer';

import cx from './IndexBuses.scss';

const ROOT_QA = 'indexBusesPage';

const IndexBuses: React.FC = () => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const location = useLocation();
    const {upperSubscriptionForm} = useExperiments();
    const needToRender = useServerDataFetcher([prefetchData]);

    const {fromSlug, toSlug, focus} = getIndexQuery(location);

    const {
        response: {crossLinks},
        project,
        isFetched: isRecipesFetched,
    } = useSelector(crossLinksSelector);

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    useEffect(() => {
        if (fromSlug || toSlug) {
            reachGoal(EBusesGoal.SEARCH_OUTDATED);
        }
    }, [fromSlug, toSlug]);

    useEffect(() => {
        if (project !== EProjectName.BUSES || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.BUSES));
        }
    }, [dispatch, isRecipesFetched, project]);

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.BUSES;

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <SearchForm
                className={cx('searchForm')}
                needToSetFromSuggestByGeoPosition
                initiallyCalendarIsOpen={Boolean(fromSlug || toSlug)}
                autoFocusFieldName={focus as ESearchFormFieldName}
            />

            {deviceType.isDesktop && (
                <PreviousSearches
                    className={cx('previousSearches')}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'previousSearches',
                    })}
                />
            )}

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            {upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Bus}
                />
            )}

            <Heading
                className={cx('title')}
                tag="h2"
                level={deviceType.isMobile ? 2 : 1}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'title',
                })}
            >
                {i18nBusesIndexBlock.title()}
            </Heading>

            <BusesAdvantages
                className={cx('advantages')}
                {...prepareQaAttributes(ROOT_QA)}
            />

            <IndexDirect
                className={cx('direct')}
                projectName={EProjectName.BUSES}
            />

            <HowToBuy
                className={cx('howToBuyATicket')}
                title={i18nBusesIndexBlock.howToBuyATicketDashTitle()}
                items={BUSES_HOW_TO_BUY_STEPS}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'howToBuyATicket',
                })}
            />

            {renderCrossaleBlock && (
                <FeaturedDirectionsBlock
                    blockClassName={cx('featureBlock')}
                    projectName={project}
                    crossLinks={crossLinks}
                />
            )}

            {!upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Bus}
                />
            )}
        </div>
    );
};

export default withReducers([
    [ELoadableReducer.BUSES_SEARCH_FORM, searchFormReducer],
    [ELoadableReducer.BUSES_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.BUSES_CALENDAR, calendarReducer],
    [ELoadableReducer.BUSES_PREVIOUS_SEARCHES, previousSearchesReducer],
    [ELoadableReducer.BUSES_CONTEXT, contextReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
])(withSaga([requestPreviousSearchesSaga])(IndexBuses));
