import React, {useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {batchActions} from 'redux-batched-actions';
import moment from 'moment-timezone';

import {STORED_SEARCHES_KEY} from 'projects/index/IndexApp/blocks/IndexBuses/constants';

import {IWithClassName} from 'types/withClassName';
import {IBusesStoredSearch} from 'types/buses/common/IBusesStoredSearch';
import {IBusesPreviousSearch} from 'types/buses/common/IBusesPreviousSearch';

import {requestPreviousSearchesAction} from 'reducers/buses/previousSearches/actions';
import {
    setBusesSearchFormFromFieldAction,
    setBusesSearchFormStartDateFieldAction,
    setBusesSearchFormToFieldAction,
} from 'reducers/buses/searchForm/actions';

import previousSearchesSelector from 'selectors/buses/previousSearches/previousSearchesSelector';

import isActualStartDate from 'sagas/index/buses/previousSearches/utilities/isActualStartDate';

import history from 'utilities/browserHistory/browserHistory';
import getSearchUrl from 'projects/buses/utilities/urls/getSearchUrl';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getNow} from 'utilities/dateUtils';

import Flex from 'components/Flex/Flex';
import PreviousSearch from 'projects/index/IndexApp/blocks/IndexBuses/components/PreviousSearches/components/PreviousSearch/PreviousSearch';
import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

interface IPreviousSearchesProps extends IWithClassName, IWithQaAttributes {}

const PreviousSearches: React.FC<IPreviousSearchesProps> = props => {
    const {previousSearches} = useSelector(previousSearchesSelector);

    const dispatch = useDispatch();

    const handleClick = useCallback(
        (item: IBusesPreviousSearch) => {
            if (!item.when) {
                return;
            }

            dispatch(
                batchActions([
                    setBusesSearchFormFromFieldAction({
                        inputValue: item.fromSuggest.title,
                        source: ESuggestSource.PRESETS,
                        selectedValue: item.fromSuggest,
                    }),
                    setBusesSearchFormToFieldAction({
                        inputValue: item.toSuggest.title,
                        source: ESuggestSource.PRESETS,
                        selectedValue: item.toSuggest,
                    }),
                    setBusesSearchFormStartDateFieldAction(item.when),
                ]),
            );

            history?.push(
                getSearchUrl({
                    fromSlug: item.fromSuggest.slug,
                    toSlug: item.toSuggest.slug,
                    when: item.when,
                }),
            );
        },
        [dispatch],
    );

    useEffect(() => {
        let storedSearches: IBusesStoredSearch[] = [];

        try {
            storedSearches = JSON.parse(
                localStorage.getItem(STORED_SEARCHES_KEY) || '[]',
            );

            const nowMoment = moment(getNow());
            const actualSearches = storedSearches.filter(({when}) =>
                isActualStartDate(when, nowMoment),
            );

            dispatch(requestPreviousSearchesAction(actualSearches));
        } catch (err) {}
    }, [dispatch]);

    if (!previousSearches.length) {
        return null;
    }

    return (
        <Flex
            between={2}
            above={6}
            inline
            nowrap
            {...prepareQaAttributes(props)}
        >
            {previousSearches.map(({value, isLoading}) => (
                <PreviousSearch
                    key={value.title}
                    item={value}
                    isLoading={isLoading}
                    onClick={handleClick}
                />
            ))}
        </Flex>
    );
};

export default React.memo(PreviousSearches);
