import React, {useEffect, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useLocation} from 'react-router-dom';
import {Location} from 'history';

import {EProjectName} from 'constants/common';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {IHotelsSuggest} from 'types/hotels/common/ISearchSuggest';
import {EIndexGoal} from 'utilities/metrika/types/goals';

import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import searchSuggestsReducer from 'reducers/depreacted/hotels/searchSuggests/reducer';
import previousSearchesReducer from 'reducers/depreacted/hotels/previousSearches/reducer';
import searchInformationReducer from 'reducers/depreacted/hotels/searchInformation/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import nonceSelector from 'selectors/common/nonceSelector';
import crossLinksSelector from 'selectors/common/crossLinksSelector';

import {deviceMods} from 'utilities/stylesUtils';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {loadable} from 'utilities/componentLoadable';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {reachGoal} from 'utilities/metrika';

import withReducers from 'containers/withReducers/withReducers';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import ChunksPreloader from 'components/ChunksPreloader/ChunksPreloader';
import FeaturedDirectionsBlock from 'components/FeaturedDirectionsBlock/FeaturedDirectionsBlock';
import SearchForm from 'projects/depreacted/hotels/components/SearchForm/SearchForm';
import IndexDirect from 'projects/index/IndexApp/components/IndexDirect/IndexDirect';
import HotelAdvantages from 'projects/depreacted/hotels/components/HotelAdvantages/HotelAdvantages';
import GeolocationError from 'projects/depreacted/hotels/components/GeolocationError/GeolocationError';
import PreviousSearches from './components/PreviousSearches/PreviousSearches';
import HotelPopularDestinations from './components/HotelPopularDestinations/HotelPopularDestinations';
import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';

import SubscriptionContainer from 'projects/index/IndexApp/containers/SubscriptionContainer/SubscriptionContainer';
import HotelsSearchInformationProvider from 'projects/depreacted/hotels/containers/HotelsSearchInformationProvider/HotelsSearchInformationProvider';

/* Init styles */
import cx from './IndexHotels.scss';

/* SSR prefetch data */
async function fetchIndexHotelsInfoSSR(): Promise<void> {}

/* Constants */
const PRELOAD_CHUNKS = [
    loadable(
        () => import('projects/depreacted/hotels/pages/SearchPage/SearchPageContainer'),
    ),
];

function getAutoFocusFieldName(location: Location): string | undefined {
    const query = getQueryByLocation(location);
    const focus = query.focus;

    if (focus === undefined || focus === null) {
        return undefined;
    }

    if (Array.isArray(focus)) {
        return focus[0];
    }

    return focus;
}

const IndexHotels: React.FC = () => {
    const {
        response: {crossLinks, recipes},
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = useSelector(crossLinksSelector);

    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const location = useLocation();
    const nonce = useSelector(nonceSelector);

    const needToRender = useServerDataFetcher([fetchIndexHotelsInfoSSR]);

    const {
        hotelsPreviousSearches: hotelsPreviousSearchesExp,
        upperSubscriptionForm,
    } = useExperiments();

    const searchFormFieldsRef = useRef<IFormFieldsRef>();

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    const onChangeToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IHotelsSuggest>) => {
            if (!fieldValue.inputValue) {
                reachGoal(EIndexGoal.HOTELS_MORDA_SEARCH_FORM_CLEAR_TO);
            }
        },
    );

    useEffect(() => {
        if (project !== EProjectName.HOTELS || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.HOTELS));
        }
    }, [dispatch, isRecipesFetched, project]);

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.HOTELS;

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            <HotelsSearchInformationProvider />
            <ChunksPreloader
                preloadChunks={PRELOAD_CHUNKS}
                maxProportionTimeout={15000}
            />

            <SearchForm
                className={cx('searchForm')}
                autoFocusFieldName={
                    getAutoFocusFieldName(location) as ESearchFormFieldName
                }
                fieldsRef={searchFormFieldsRef}
                onChangeToPoint={onChangeToPoint}
            />

            {hotelsPreviousSearchesExp && deviceType.isDesktop && (
                <PreviousSearches
                    className={cx('previousSearches')}
                    searchFormFieldsRef={searchFormFieldsRef}
                />
            )}

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            {upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Hotels}
                />
            )}

            <HotelAdvantages className={cx('advantages')} />

            {renderCrossaleBlock && (
                <HotelPopularDestinations
                    className={cx('recipes')}
                    recipes={recipes}
                    isLoading={isRecipesLoading}
                    deviceType={deviceType}
                    nonce={nonce}
                />
            )}

            <IndexDirect
                className={cx('direct')}
                projectName={EProjectName.HOTELS}
            />

            {renderCrossaleBlock && (
                <FeaturedDirectionsBlock
                    blockClassName={cx('featureBlock')}
                    projectName={project}
                    crossLinks={crossLinks}
                />
            )}

            {!upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Hotels}
                />
            )}

            <GeolocationError />
        </div>
    );
};

export default withReducers([
    [ELoadableReducer.HOTELS_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.HOTELS_SEARCH_INFORMATION, searchInformationReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
    [ELoadableReducer.HOTELS_PREVIOUS_SEARCHES, previousSearchesReducer],
])(IndexHotels);
