import React, {PureComponent, ReactNode} from 'react';
import _times from 'lodash/times';
import {withRouter, RouteComponentProps} from 'react-router-dom';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IDirection, IRecipeBlock} from 'server/api/CommonApi/types/CrossLinks';

/* Utilities */
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getCrossLinksUrl from 'utilities/url/getCrossLinksUrl';
import getCrossLinkImageUrlByType from 'utilities/url/getCrossLinkImageUrlByType';
import isLazyLoadingRecipeImage from 'projects/index/IndexApp/utilities/isLazyLoadingRecipeImage';

import * as i18nBlock from 'i18nNew/hotels-Recipes';

/* Components */
import Heading from 'components/Heading/Heading';
import Recipe, {TRecipe} from 'components/Recipe/Recipe';
import RecipeGrid from 'components/RecipeGrid/RecipeGrid';
import RecipeSkeleton from 'components/RecipeSkeleton/RecipeSkeleton';
import SchemaMarkup, {
    getAggregateOffer,
} from 'components/SchemaMarkup/SchemaMarkup';

/* Init styles */
import cx from './HotelPopularDestinations.scss';

/* Constants */
const IMAGE_SIZE = 'orig';
const TOTAL_SKELETON_COUNT = 5;
const SKELETON_DATA = _times<TSkeletonItemType>(
    TOTAL_SKELETON_COUNT,
    () => ({}),
);

/* Components Types */
export interface IHotelPopularDestinationsProps
    extends RouteComponentProps,
        IWithDeviceType,
        IWithClassName {
    isLoading: boolean;
    recipes: IRecipeBlock;
    nonce: string;
}
type TSkeletonItemType = {};

class HotelPopularDestinations extends PureComponent<IHotelPopularDestinationsProps> {
    private getPopularDestinationUrl = (recipe: IDirection): string => {
        return getCrossLinksUrl({
            type: EProjectName.HOTELS,
            fromSlug: recipe?.from?.slug,
            toSlug: recipe.to.slug,
        });
    };

    private renderSkeletonDestination = (
        data: TSkeletonItemType,
        type: TRecipe,
        index: number,
    ): ReactNode => {
        return <RecipeSkeleton key={index} type={type} />;
    };

    private renderMarkup(recipes: IDirection[]): React.ReactNode {
        const {nonce} = this.props;

        return recipes.map((recipe, i) => {
            if (!recipe.minPrice) {
                return null;
            }

            const url = getCrossLinksUrl({
                type: EProjectName.AVIA,
                fromSlug: recipe?.from?.slug,
                toSlug: recipe.to.slug,
            });

            const data = {
                name: i18nBlock.blockDashTitleDashNew(),
                price: recipe.minPrice,
                url,
            };

            return (
                <SchemaMarkup
                    data={getAggregateOffer(data)}
                    key={i}
                    nonce={nonce}
                />
            );
        });
    }

    private renderDestination = (
        data: IDirection,
        type: TRecipe,
        idx: number,
    ): ReactNode => {
        const {recipes, deviceType} = this.props;
        const {to, imageUrl} = data;
        const popularDestinationUrl = this.getPopularDestinationUrl(data);
        const preparedImage = imageUrl && imageUrl.replace(IMAGE_SIZE, '');

        return (
            <Recipe
                key={idx}
                href={popularDestinationUrl}
                target="_blank"
                rel="noopener noreferrer"
                targetCity={to.linguistics.nominativeCase}
                imageUrl={getCrossLinkImageUrlByType(preparedImage, type)}
                type={type}
                isLazy={isLazyLoadingRecipeImage(
                    recipes.directions.length,
                    idx,
                    deviceType.isMobile,
                )}
                {...prepareQaAttributes({key: idx, current: 'popular-place'})}
            />
        );
    };

    render(): JSX.Element {
        const {deviceType, recipes, isLoading, className} = this.props;
        const isMobile = deviceType.isMobile;
        const qaAttribute = 'popular-places';

        return (
            <section
                className={className}
                {...prepareQaAttributes(qaAttribute)}
            >
                <Heading
                    className={cx('title')}
                    level={isMobile ? 2 : 1}
                    tag="h2"
                    {...prepareQaAttributes({
                        parent: qaAttribute,
                        current: 'title',
                    })}
                >
                    {i18nBlock.blockDashTitleDashNew()}
                </Heading>

                {isLoading || recipes.directions.length === 0 ? (
                    <RecipeGrid
                        deviceType={deviceType}
                        recipesData={SKELETON_DATA}
                        renderRecipe={this.renderSkeletonDestination}
                    />
                ) : (
                    <RecipeGrid
                        deviceType={deviceType}
                        recipesData={recipes.directions}
                        renderRecipe={this.renderDestination}
                    />
                )}
                {this.renderMarkup(this.props.recipes.directions)}
            </section>
        );
    }
}

export default withRouter(HotelPopularDestinations);
