import React, {MutableRefObject, useCallback, useMemo} from 'react';
import {useHistory} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {ESearchFormFieldName} from 'components/SearchForm/types';
import {IHotelsPreviousSearch} from 'types/hotels/previousSearches/IPreviousSearch';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import fillSearchFormByPreviousSearchThunkAction from 'reducers/depreacted/hotels/previousSearches/thunk/fillSearchFormByPreviousSearch';

import {hotelsURLs} from 'projects/depreacted/hotels/utilities/urls';
import {reachGoal} from 'utilities/metrika';
import getPreviousSearchDateDescription from 'projects/depreacted/hotels/utilities/previousSearches/getPreviousSearchDateDescription';

import * as i18n from 'i18nNew/common';

import SearchPreset from 'components/SearchPreset/SearchPreset';
import Text from 'components/Text/Text';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import HotelIcon from 'icons/24/Hotel';
import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

interface IPreviousSearchProps {
    item: IHotelsPreviousSearch;
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined>;
}

const PreviousSearch: React.FC<IPreviousSearchProps> = props => {
    const {item, searchFormFieldsRef} = props;
    const history = useHistory();
    const dispatch = useDispatch();

    const handleClick = useCallback(() => {
        dispatch(fillSearchFormByPreviousSearchThunkAction(item));

        if (item.startDate) {
            history?.push(
                hotelsURLs.getSearchUrl(
                    {
                        searchFormFieldsValues: {
                            [ESearchFormFieldName.TO]: {
                                selectedValue: item.to,
                                inputValue: item.to.name,
                                source: ESuggestSource.PRESETS,
                            },
                            [ESearchFormFieldName.START_DATE]: item.startDate,
                            [ESearchFormFieldName.END_DATE]: item.endDate,
                            adults: item.adults,
                            childrenAges: item.childrenAges,
                        },
                    },
                    {
                        canSearchOnTravel: true,
                        hasProductionOffers: false,
                        hasDirectionSuggest: true,
                    },
                ),
            );
        } else {
            searchFormFieldsRef.current?.focusFieldByName(
                ESearchFormFieldName.START_DATE,
            );
        }

        reachGoal(EHotelsGoal.PREVIOUS_SEARCHES_PRESET_CLICK);
    }, [dispatch, item, searchFormFieldsRef, history]);

    const secondRow = useMemo(() => {
        return (
            getPreviousSearchDateDescription(item) || (
                <Text size="s" color="secondary">
                    {i18n.selectDates()}
                </Text>
            )
        );
    }, [item]);

    return (
        <SearchPreset
            onClick={handleClick}
            title={item.to.name}
            secondRow={secondRow}
            Icon={HotelIcon}
        />
    );
};

export default React.memo(PreviousSearch);
