import React, {useCallback, useMemo} from 'react';
import {ParsedQuery} from 'query-string';

import {EProjectName} from 'constants/common';

import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getTabsManifest from 'projects/index/IndexApp/blocks/IndexTabs/utilities/getTabsManifest';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useSingleProjectView} from 'utilities/hooks/useSingleProjectView';
import {useOnNavigateToProject} from 'utilities/hooks/useOnNavigateToProject/useOnNavigateToProject';

import TabLink from 'projects/index/IndexApp/blocks/IndexTabs/components/TabLink/TabLink';
import HorizontalHiddenScroll from 'components/HorizontalHiddenScroll/HorizontalHiddenScroll';

import cx from './IndexTabs.scss';

interface IIndexTabsProps {
    tabs: EProjectName[];
    activeTabId: EProjectName;
    query: ParsedQuery;
    isWeb: boolean;
}

const ROOT_QA = 'indexTabs';

const IndexTabs: React.FC<IIndexTabsProps> = props => {
    const deviceType = useDeviceType();

    const {query, tabs, activeTabId, isWeb} = props;

    const isSingleProjectView = useSingleProjectView();
    const areTabsVisible = isWeb && !isSingleProjectView;
    const onNavigateToProject = useOnNavigateToProject();

    const tabsManifest = useMemo(() => getTabsManifest(), []);
    const Container = deviceType.isMobile ? HorizontalHiddenScroll : 'div';

    const handleChangeActiveTab = useCallback(
        (tabId: EProjectName): void => {
            if (tabsManifest[tabId].metrikaId) {
                reachGoal(tabsManifest[tabId].metrikaId);
            }
        },
        [tabsManifest],
    );

    const renderTabsContent = useCallback(
        (tabId: EProjectName) => {
            const {tabContent: TabContent} = tabsManifest[tabId];

            return <TabContent query={query} />;
        },
        [tabsManifest, query],
    );

    const tabsBlock = useMemo(() => {
        return (
            <>
                {areTabsVisible && (
                    <div className={cx('tabsWrapper')}>
                        <Container className={cx('tabsContainer')}>
                            {tabs.map(tabId => {
                                const isActive = activeTabId === tabId;

                                const {
                                    tabTitle,
                                    tabLink,
                                    icon: Icon,
                                } = tabsManifest[tabId];

                                return (
                                    <TabLink
                                        key={tabId}
                                        isActive={isActive}
                                        tabLink={tabLink}
                                        Icon={Icon}
                                        title={tabTitle}
                                        onClick={(): void => {
                                            handleChangeActiveTab(tabId);
                                            onNavigateToProject(tabId);
                                        }}
                                        {...prepareQaAttributes({
                                            parent: ROOT_QA,
                                            current: 'indexTab',
                                            key: tabId,
                                        })}
                                    />
                                );
                            })}
                        </Container>
                    </div>
                )}
                {renderTabsContent(activeTabId)}
            </>
        );
    }, [
        areTabsVisible,
        Container,
        tabs,
        renderTabsContent,
        activeTabId,
        tabsManifest,
        handleChangeActiveTab,
        onNavigateToProject,
    ]);

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(ROOT_QA)}
        >
            {tabsBlock}
        </div>
    );
};

export default IndexTabs;
