import React, {useEffect, useRef} from 'react';
import {ParsedQuery} from 'query-string';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {TRAINS_HOW_TO_BUY_STEPS} from 'projects/index/IndexApp/blocks/IndexTrains/constants/howToBuySteps';
import {URLs} from 'constants/urls';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import crossLinksSelector from 'selectors/common/crossLinksSelector';
import nonceSelector from 'selectors/common/nonceSelector';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nTrainsIndexBlock from 'i18nNew/trains-indexTexts';

import withReducers from 'containers/withReducers/withReducers';

import IndexDirect from 'projects/index/IndexApp/components/IndexDirect/IndexDirect';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import TrainsAdvantages from 'projects/trains/components/TrainsAdvantages/TrainsAdvantages';
import HowToBuy from 'components/HowToBuy/HowToBuy';
import FeaturedDirectionsBlock from 'components/FeaturedDirectionsBlock/FeaturedDirectionsBlock';
import TrainsIndexTexts from 'projects/index/IndexApp/blocks/IndexTrains/components/TrainsIndexTexts/TrainsIndexTexts';
import TrainsPopularDirections from 'projects/index/IndexApp/blocks/IndexTrains/components/TrainsPopularDirections/TrainsPopularDirections';
import PreviousSearches from 'projects/index/IndexApp/blocks/IndexTrains/components/PreviousSearches/PreviousSearches';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';

import SubscriptionContainer from 'projects/index/IndexApp/containers/SubscriptionContainer/SubscriptionContainer';

import cx from './IndexTrains.scss';

const ROOT_QA = 'indexTrainsPage';

async function prefetchData(): Promise<void> {
    await Promise.all([]);
}

interface IIndexTrainsProps {
    query?: ParsedQuery;
}

const IndexTrains: React.FC<IIndexTrainsProps> = () => {
    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const {
        response: {crossLinks, recipes},
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = useSelector(crossLinksSelector);
    const nonce = useSelector(nonceSelector);
    const searchFormFieldsRef = useRef<IFormFieldsRef>();
    const {upperSubscriptionForm} = useExperiments();

    const needToRender = useServerDataFetcher([prefetchData]);

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    useEffect(() => {
        if (project !== EProjectName.TRAINS || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.TRAINS));
        }
    }, [dispatch, isRecipesFetched, project]);

    // Параметры m__outdate_search_request и m__outdate_order_request появляются в query при редиректе
    // cо страниц поиска и заказа с датами в прошлом.
    // query.m__outdate_index_request - при редиректе с главной страницы, если в query есть дата в прошлом.

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.TRAINS;

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            {deviceType.isDesktop && (
                <PreviousSearches
                    className={cx('previousSearches')}
                    searchFormFieldsRef={searchFormFieldsRef}
                />
            )}

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            {upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Trains}
                />
            )}

            <TrainsAdvantages className={cx('advantages')} />

            {renderCrossaleBlock && (
                <TrainsPopularDirections
                    className={cx('recipes')}
                    recipes={recipes}
                    deviceType={deviceType}
                    isLoading={isRecipesLoading}
                    nonce={nonce}
                />
            )}

            <IndexDirect
                className={cx('direct')}
                projectName={EProjectName.TRAINS}
            />

            <HowToBuy
                className={cx('howToBuyATicket')}
                title={i18nTrainsIndexBlock.howToBuyATicketDashTitleDash()}
                items={TRAINS_HOW_TO_BUY_STEPS}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'howToBuyATicket',
                })}
            />

            <TrainsIndexTexts
                className={cx('texts')}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'texts',
                })}
            />

            {renderCrossaleBlock && (
                <FeaturedDirectionsBlock
                    blockClassName={cx('featureBlock')}
                    crossLinks={crossLinks}
                    projectName={project}
                    morePopularUrl={URLs.trainsPopularRoutes}
                />
            )}

            {!upperSubscriptionForm && (
                <SubscriptionContainer
                    className={cx('subscribe')}
                    vertical={ESubscriptionVerticalName.Trains}
                />
            )}
        </div>
    );
};

export default withReducers([
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
])(IndexTrains);
