import React, {useCallback, useMemo} from 'react';

import {EProjectName} from 'constants/common';

import IPrice from 'types/common/price/IPrice';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IDirection, IRecipeBlock} from 'server/api/CommonApi/types/CrossLinks';

import getCrossLinksUrl from 'utilities/url/getCrossLinksUrl';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getCrossLinkImageUrlByType from 'utilities/url/getCrossLinkImageUrlByType';
import isLazyLoadingRecipeImage from 'projects/index/IndexApp/utilities/isLazyLoadingRecipeImage';

import * as i18nBlock from 'i18nNew/trains-popularDirections';

import Price from 'components/Price/Price';
import Heading from 'components/Heading/Heading';
import Recipe, {TRecipe} from 'components/Recipe/Recipe';
import RecipeGrid from 'components/RecipeGrid/RecipeGrid';
import RecipeSkeleton from 'components/RecipeSkeleton/RecipeSkeleton';
import SchemaMarkup, {
    getAggregateOffer,
} from 'components/SchemaMarkup/SchemaMarkup';

import cx from './TrainsPopularDirections.scss';

interface ITrainsPopularDirectionsPropTypes
    extends IWithDeviceType,
        IWithClassName {
    recipes: IRecipeBlock;
    isLoading: boolean;
    nonce: string;
}

const DEFAULT_SKELETON_DATA = Array(5).fill({});

const ROOT_QA = 'popular-directions';

/**
 * Компонент для отображения популярных направлений поездов на главной странице портала.
 */
const TrainsPopularDirections: React.FC<ITrainsPopularDirectionsPropTypes> =
    props => {
        const {deviceType, recipes, nonce, isLoading, className} = props;
        const isMobile = deviceType.isMobile;

        const renderSkeletonRecipe = useCallback(
            (
                receiptData: null,
                type: TRecipe,
                index: number,
            ): React.ReactNode => {
                return <RecipeSkeleton key={index} type={type} />;
            },
            [],
        );

        const renderPrice = (
            qa: IWithQaAttributes,
            minPrice?: IPrice,
        ): React.ReactNode => {
            if (!minPrice) {
                return null;
            }

            return (
                <Price
                    isFrom
                    isRound
                    value={minPrice.value}
                    currency={minPrice.currency}
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'price',
                    })}
                />
            );
        };

        const renderPopularDirectionRecipe = (
            receiptData: IDirection,
            type: TRecipe,
            idx: number,
        ): React.ReactNode => {
            const key = `${receiptData?.from?.slug}-${receiptData.to.slug}`;
            const qaAttr = prepareQaAttributes({
                key,
                current: 'trains-popular-direction',
            });
            const price = renderPrice(qaAttr, receiptData.minPrice);

            const searchLink = getCrossLinksUrl({
                type: EProjectName.TRAINS,
                fromSlug: receiptData?.from?.slug,
                toSlug: receiptData.to.slug,
            });

            return (
                <Recipe
                    key={key}
                    to={searchLink}
                    targetCountry={i18nBlock.departureFrom({
                        from:
                            receiptData?.from?.linguistics?.genitiveCase || '',
                    })}
                    targetCity={i18nBlock.departureTo({
                        to: receiptData.to.linguistics.accusativeCase,
                    })}
                    price={price}
                    imageUrl={getCrossLinkImageUrlByType(
                        receiptData.imageUrl,
                        type,
                    )}
                    type={type}
                    isLazy={isLazyLoadingRecipeImage(
                        recipes.directions.length,
                        idx,
                        deviceType.isMobile,
                    )}
                    {...qaAttr}
                />
            );
        };

        const markup = useMemo(() => {
            return recipes.directions.map((item, i) => {
                if (!item.minPrice) return null;

                const url = getCrossLinksUrl({
                    type: EProjectName.TRAINS,
                    fromSlug: item?.from?.slug,
                    toSlug: item.to.slug,
                });

                const data = {
                    name: i18nBlock.markupDashText(),
                    price: item.minPrice,
                    url,
                };

                return (
                    <SchemaMarkup
                        data={getAggregateOffer(data)}
                        key={i}
                        nonce={nonce}
                    />
                );
            });
        }, [recipes, nonce]);

        const itemsCount = recipes.directions.length;

        if (!isLoading && itemsCount === 0) {
            return null;
        }

        return (
            <section className={className} {...prepareQaAttributes(ROOT_QA)}>
                <Heading
                    className={cx('title')}
                    level={isMobile ? 2 : 1}
                    tag="h2"
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'title',
                    })}
                >
                    {i18nBlock.blockDashTitle()}
                </Heading>

                {itemsCount === 0 ? (
                    <RecipeGrid
                        deviceType={deviceType}
                        recipesData={DEFAULT_SKELETON_DATA}
                        renderRecipe={renderSkeletonRecipe}
                    />
                ) : (
                    <RecipeGrid
                        deviceType={deviceType}
                        recipesData={recipes.directions}
                        renderRecipe={renderPopularDirectionRecipe}
                    />
                )}

                {markup}
            </section>
        );
    };

export default TrainsPopularDirections;
