import {createReducer} from 'typesafe-actions';

import {
    OFFER_FETCHED,
    OFFER_IS_NOT_AVAILABLE,
    UNKNOWN_OFFER_ERROR,
    OFFER_NOT_FOUND_ERROR,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';

import {
    IBookOffer,
    TBookOfferError,
    IBookOfferInfoError,
    IBookOfferNotAvaliableError,
    IBookOfferNotFoundError,
    IParsedTokenInfo,
} from './types';

import {
    TOfferInfoByTokenActionTypes,
    fetchOfferInfoByTokenAction,
    updateOfferInfoAction,
    resetOfferInfoAction,
} from './actions';
import {stopOrderInfoPollingAction} from '../orderInfo/actions';

export interface IOfferInfoByTokenReducer {
    status: string;
    isError?: boolean;
    isSuccess: boolean;
    isLoading: boolean;
    offerInfo?: IBookOffer;
    error?: IBookOfferInfoError;
    tokenInfo?: IParsedTokenInfo;
}

const initialState: IOfferInfoByTokenReducer = {
    status: '',
    isError: false,
    isSuccess: false,
    isLoading: false,
};

function isNotAvaliableError(
    value: TBookOfferError,
): value is IBookOfferNotAvaliableError {
    return value.code === OFFER_IS_NOT_AVAILABLE;
}

function isNotFoundError(
    value: TBookOfferError,
): value is IBookOfferNotFoundError {
    return value.code === OFFER_NOT_FOUND_ERROR;
}

export default createReducer<
    IOfferInfoByTokenReducer,
    TOfferInfoByTokenActionTypes
>(initialState)
    .handleAction(fetchOfferInfoByTokenAction.request, () => ({
        ...initialState,
        isLoading: true,
    }))
    .handleAction(fetchOfferInfoByTokenAction.success, (state, {payload}) => {
        const {data} = payload;

        if (data) {
            /* Success offer */

            return {
                ...state,
                isSuccess: true,
                isLoading: false,
                status: OFFER_FETCHED,
                offerInfo: data,
            };
        }

        return {
            ...initialState,
            status: UNKNOWN_OFFER_ERROR,
            isError: true,
        };
    })
    .handleAction(fetchOfferInfoByTokenAction.failure, (_, {payload}) => {
        const {error} = payload;
        const {code: errorCode} = error || {};

        if (isNotAvaliableError(error) && error.offerInfo) {
            /* Fetched expired offer */

            return {
                status: OFFER_IS_NOT_AVAILABLE,
                isLoading: false,
                isSuccess: true,
                offerInfo: error.offerInfo,
            };
        }

        if (isNotFoundError(error) && error.tokenInfo) {
            /* Offer not found in cache */

            return {
                ...initialState,
                status: OFFER_NOT_FOUND_ERROR,
                isLoading: false,
                tokenInfo: error.tokenInfo,
            };
        }

        /* Fetched offer with errorCode */
        return {
            ...initialState,
            status: errorCode || UNKNOWN_OFFER_ERROR,
            isError: true,
        };
    })
    .handleAction(updateOfferInfoAction, (state, {payload}) => {
        const {data} = payload;

        return {
            ...state,
            offerInfo: data,
        };
    })
    .handleAction(stopOrderInfoPollingAction, state => ({
        ...state,
        isLoading: false,
    }))
    .handleAction(resetOfferInfoAction, () => initialState);
