import {createReducer} from 'typesafe-actions';
import omit from 'lodash/omit';

/* Types */
import {
    IFullFilters,
    TCollapsedFilters,
} from 'types/hotels/search/IFiltersInfo';
import {IResetFilterInfo} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {
    fillFiltersBySearchInfo,
    getUpdatedFiltersRangePrice,
    getCurrentFiltersAfterReset,
    getPermanentAndCurrentFilters,
    getUpdatedFiltersByActiveAtoms,
    getUpdatedFiltersByResetAction,
} from './utilities';
import collapsedFiltersSaver from 'projects/depreacted/hotels/utilities/filters/collapsedFiltersSaver';

import {
    fillFilters,
    FiltersActionsType,
    changeFilterGroup,
    applyFilters,
    revertFilters,
    changePriceFilter,
    resetFilters,
    resetFiltersToInitial,
    updateCountFiltersActions,
    toggleFilterCollapse,
    fillResetActions,
    resetFilterAction,
} from './actions';

export interface IHotelsFiltersReducer {
    needSyncPermanentFiltersWithServer: boolean;
    needSyncCurrentFiltersWithServer: boolean;
    permanentFilters: IFullFilters;
    currentFilters: IFullFilters;
    collapsedFilters: TCollapsedFilters;
    resetFilterInfo: IResetFilterInfo | null;
    resetFilterWasUsed: boolean;
}

/* Initial State */

export const initialFilters: IFullFilters = {
    activeFilterAtoms: {},
};

const initialState: IHotelsFiltersReducer = {
    needSyncPermanentFiltersWithServer: false,
    needSyncCurrentFiltersWithServer: false,
    permanentFilters: initialFilters,
    currentFilters: initialFilters,
    collapsedFilters: collapsedFiltersSaver.getFromStorage(),
    resetFilterInfo: null,
    resetFilterWasUsed: false,
};

/* Reducer */
export default createReducer<IHotelsFiltersReducer, FiltersActionsType>(
    initialState,
)
    /* Fill filters by searchInfo from server */
    .handleAction(fillFilters, (state, {payload}): IHotelsFiltersReducer => {
        const currentAndPermanentFilters = fillFiltersBySearchInfo(payload);

        return {
            ...state,
            ...currentAndPermanentFilters,
            needSyncPermanentFiltersWithServer: false,
            needSyncCurrentFiltersWithServer: false,
        };
    })
    /* Fill reset actions by searchInfo from server */
    .handleAction(
        fillResetActions,
        (state, {payload}): IHotelsFiltersReducer => {
            return {
                ...state,
                resetFilterInfo: payload,
                resetFilterWasUsed: false,
            };
        },
    )
    /* Reset filter pressed */
    .handleAction(
        resetFilterAction,
        (state, {payload}): IHotelsFiltersReducer => {
            const activeFiltersAtomsAfterReset = getUpdatedFiltersByResetAction(
                payload,
                state.currentFilters,
            );

            return {
                ...state,
                currentFilters: activeFiltersAtomsAfterReset,
                permanentFilters: activeFiltersAtomsAfterReset,
                needSyncPermanentFiltersWithServer: true,
                needSyncCurrentFiltersWithServer: true,
                resetFilterWasUsed: true,
            };
        },
    )
    /* Change FilterActiveAtoms */
    .handleAction(
        changeFilterGroup,
        (state, {payload}): IHotelsFiltersReducer => {
            const {targetFiltersType} = payload;
            const getUpdatedFilters = getUpdatedFiltersByActiveAtoms(payload);

            return getPermanentAndCurrentFilters({
                state,
                targetFiltersType,
                getUpdatedFilters,
            });
        },
    )
    /* Change PriceFilter value */
    .handleAction(
        changePriceFilter,
        (state, {payload}): IHotelsFiltersReducer => {
            const {targetFiltersType} = payload;
            const getUpdatedFilters = getUpdatedFiltersRangePrice(payload);

            return getPermanentAndCurrentFilters({
                state,
                targetFiltersType,
                getUpdatedFilters,
            });
        },
    )
    /* Apply and start sync PermanentFilters with server */
    .handleAction(applyFilters, (state): IHotelsFiltersReducer => {
        const {currentFilters} = state;

        return {
            ...state,
            needSyncPermanentFiltersWithServer: true,
            permanentFilters: currentFilters,
        };
    })
    /* Success sync CurrentFilters with server and update hotelCounts */
    .handleAction(
        updateCountFiltersActions.success,
        (state, {payload}): IHotelsFiltersReducer => {
            const {filterInfo, searchControlInfo, foundHotelCount} = payload;
            const {currentFilters} = state;
            const {priceFilter, activeFilterAtoms} = currentFilters;
            const {quickFilters, detailedFilters, detailedFiltersBatches} =
                filterInfo;
            const {quickControls = quickFilters} = searchControlInfo || {};

            return {
                ...state,
                needSyncCurrentFiltersWithServer: false,
                currentFilters: {
                    quickFilters,
                    quickControls,
                    priceFilter,
                    activeFilterAtoms,
                    detailedFilters,
                    detailedFiltersBatches,
                    foundHotelCount,
                },
            };
        },
    )
    /* Revert CurrentFilters to PermanentFilters */
    .handleAction(
        revertFilters,
        (state): IHotelsFiltersReducer => ({
            ...state,
            needSyncCurrentFiltersWithServer: false,
            currentFilters: state.permanentFilters,
        }),
    )
    /* Reset CurrentFilters */
    .handleAction(resetFilters, (state): IHotelsFiltersReducer => {
        const {currentFilters} = state;

        return {
            ...state,
            needSyncCurrentFiltersWithServer: true,
            currentFilters: getCurrentFiltersAfterReset(currentFilters),
        };
    })
    /* Reset To Initial */
    .handleAction(
        resetFiltersToInitial,
        (): IHotelsFiltersReducer => initialState,
    )
    .handleAction(
        toggleFilterCollapse,
        (state, {payload}): IHotelsFiltersReducer => ({
            ...state,
            collapsedFilters: payload.wasCollapsed
                ? omit(state.collapsedFilters, payload.id)
                : {...state.collapsedFilters, [payload.id]: true},
        }),
    );
