import {
    ActiveFilterAtomsType,
    FilterAtomType,
    IChangeFilterGroupPayload,
    IChangeFiltersTargetType,
    IChangePriceFilterPayload,
    IFullFilters,
    IPriceFilter,
    IRangePriceFilter,
} from 'types/hotels/search/IFiltersInfo';
import {ISearchHotelsInfo} from 'server/api/HotelsSearchAPI/types/ISearchHotels';
import {IResetFilterAction} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {IHotelsFiltersReducer} from 'reducers/depreacted/hotels/searchPage/filters/reducer';

export const fillPriceFilter = ({
    priceFilter,
    filterPriceFrom,
    filterPriceTo,
}: {
    priceFilter: IPriceFilter;
    filterPriceFrom?: number;
    filterPriceTo?: number;
}): IRangePriceFilter | undefined => {
    if (priceFilter) {
        return {
            minValue: filterPriceFrom || priceFilter.minPriceEstimate,
            maxValue: filterPriceTo || priceFilter.maxPriceEstimate,
            minPriceEstimate: priceFilter.minPriceEstimate,
            maxPriceEstimate: priceFilter.maxPriceEstimate,
            currency: priceFilter.currency,
        };
    }
};

export const convertFilterAtoms = (
    filterAtoms?: FilterAtomType[],
): ActiveFilterAtomsType => {
    if (filterAtoms) {
        return filterAtoms.reduce(
            (activeFilterAtoms: ActiveFilterAtomsType, filterAtom) => {
                activeFilterAtoms[filterAtom] = true;

                return activeFilterAtoms;
            },
            {},
        );
    }

    return {};
};

export const fillFiltersBySearchInfo = (
    searchInfo: ISearchHotelsInfo,
): {
    permanentFilters: IFullFilters;
    currentFilters: IFullFilters;
} => {
    const {
        filterInfo: {
            params: {filterPriceFrom, filterPriceTo, filterAtoms},
            priceFilter,
            quickFilters,
            detailedFilters,
            detailedFiltersBatches,
        },
        searchControlInfo,
        foundHotelCount,
    } = searchInfo;
    const {quickControls = quickFilters} = searchControlInfo || {};

    const filters: IFullFilters = {
        priceFilter: fillPriceFilter({
            priceFilter,
            filterPriceFrom,
            filterPriceTo,
        }),
        activeFilterAtoms: convertFilterAtoms(filterAtoms),
        foundHotelCount,
        quickFilters,
        quickControls,
        detailedFilters,
        detailedFiltersBatches,
    };

    return {
        permanentFilters: filters,
        currentFilters: filters,
    };
};

export const getPermanentAndCurrentFilters = ({
    targetFiltersType,
    state,
    getUpdatedFilters,
}: {
    targetFiltersType: IChangeFiltersTargetType['targetFiltersType'];
    state: IHotelsFiltersReducer;
    getUpdatedFilters: (filters: IFullFilters) => IFullFilters;
}): IHotelsFiltersReducer => {
    const {currentFilters} = state;
    const updatedFilters = getUpdatedFilters(currentFilters);

    switch (targetFiltersType) {
        case 'ALL': {
            return {
                ...state,
                needSyncPermanentFiltersWithServer: true,
                permanentFilters: updatedFilters,
                currentFilters: updatedFilters,
            };
        }
        case 'CURRENT': {
            return {
                ...state,
                needSyncCurrentFiltersWithServer: true,
                currentFilters: updatedFilters,
            };
        }
    }
};

export const getUpdatedFiltersByActiveAtoms =
    (payload: IChangeFilterGroupPayload) =>
    (filters: IFullFilters): IFullFilters => {
        const {atomsOn, atomsOff} = payload;
        const {activeFilterAtoms} = filters;
        const updatedActiveFilterAtoms = {
            ...activeFilterAtoms,
        };

        if (atomsOn) {
            atomsOn.forEach((atom: FilterAtomType) => {
                updatedActiveFilterAtoms[atom] = true;
            });
        }

        if (atomsOff) {
            atomsOff.forEach((atom: FilterAtomType) => {
                delete updatedActiveFilterAtoms[atom];
            });
        }

        return {
            ...filters,
            activeFilterAtoms: updatedActiveFilterAtoms,
        };
    };

export const getUpdatedFiltersByResetAction = (
    payload: IResetFilterAction,
    filters: IFullFilters,
) => {
    let newPriceFilter = null;
    const {atomsOff, needResetFilterPrice} = payload;
    const {activeFilterAtoms} = filters;
    const updatedActiveFilterAtoms = {
        ...activeFilterAtoms,
    };

    if (needResetFilterPrice && filters.priceFilter) {
        newPriceFilter = {
            ...filters.priceFilter,
            minValue: filters.priceFilter?.minPriceEstimate,
            maxValue: filters.priceFilter?.maxPriceEstimate,
        };
    }

    if (atomsOff) {
        atomsOff.forEach((atom: FilterAtomType) => {
            delete updatedActiveFilterAtoms[atom];
        });
    }

    return {
        ...filters,
        activeFilterAtoms: updatedActiveFilterAtoms,
        ...(newPriceFilter ? {priceFilter: newPriceFilter} : {}),
    };
};

export const getUpdatedFiltersRangePrice =
    (payload: IChangePriceFilterPayload) =>
    (filters: IFullFilters): IFullFilters => {
        const {priceFilter} = filters;

        if (priceFilter) {
            return {
                ...filters,
                priceFilter: {
                    ...priceFilter,
                    ...payload,
                },
            };
        }

        return filters;
    };

export const getCurrentFiltersAfterReset = (
    currentFilters: IFullFilters,
): IFullFilters => {
    const {priceFilter} = currentFilters;

    const updatedPriceFilter = priceFilter
        ? {
              ...priceFilter,
              minValue: priceFilter.minPriceEstimate,
              maxValue: priceFilter.maxPriceEstimate,
          }
        : undefined;

    return {
        ...currentFilters,
        activeFilterAtoms: {},
        priceFilter: updatedPriceFilter,
    };
};
