import {createReducer, ActionType} from 'typesafe-actions';
import _uniq from 'lodash/uniq';

import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {MapBoundsType} from 'types/common/ICoordinates';

import * as actions from './actions';

export interface ISearchMapReducer {
    selectedMapMarkerPermalink?: PermalinkType;
    viewedMapMarkerPermalinks: Record<PermalinkType, boolean>;
    mapBounds?: MapBoundsType;
    mapBoundsBySearch?: MapBoundsType;
    hotels: PermalinkType[];
}

/* Initial State */

const INITIAL_STATE: ISearchMapReducer = {
    viewedMapMarkerPermalinks: {},
    hotels: [],
};

/* Reducer */

const searchMapReducer = createReducer<
    ISearchMapReducer,
    ActionType<typeof actions>
>(INITIAL_STATE)
    .handleAction(
        actions.clickMapMarker,
        (state, {payload}): ISearchMapReducer => ({
            ...state,
            viewedMapMarkerPermalinks: {
                ...state.viewedMapMarkerPermalinks,
                [payload]: true,
            },
        }),
    )
    .handleAction(
        actions.selectMapMarker,
        (state, {payload}): ISearchMapReducer => ({
            ...state,
            selectedMapMarkerPermalink: payload,
        }),
    )
    .handleAction(
        actions.setMapBounds,
        (state, {payload}): ISearchMapReducer => ({
            ...state,
            mapBounds: payload,
        }),
    )
    .handleAction(actions.fillMap, (state, {payload}): ISearchMapReducer => {
        const {hotels: hotelsFromState} = state;
        const {hotels: hotelsBySearch, bboxAsStruct} = payload;
        const hotelBySearchPermalinks = hotelsBySearch.map(
            ({hotel}) => hotel.permalink,
        );
        const hotels = _uniq([...hotelsFromState, ...hotelBySearchPermalinks]);

        return {
            ...state,
            hotels,
            mapBounds: bboxAsStruct,
            mapBoundsBySearch: bboxAsStruct,
        };
    })
    .handleAction(
        actions.clearMapHotels,
        (state): ISearchMapReducer => ({
            ...state,
            hotels: [],
        }),
    )
    .handleAction(
        actions.resetMapToInitial,
        (): ISearchMapReducer => INITIAL_STATE,
    );

export default searchMapReducer;
