import {batchActions} from 'redux-batched-actions';
import {takeEvery, call, put, select, race, take} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {
    UNKNOWN_OFFER_ERROR,
    UNKNOWN_ORDER_ERROR,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {
    TCreateOrderResponse,
    IOrderError,
    TCreateOrderErrorResponse,
    ICreateOrderRequest,
} from 'types/hotels/book/ICreateOrder';
import {
    IAppliedPromoCodesInfo,
    EPromoCodesApplicationType,
} from 'types/hotels/book/IApplyPromoCodes';

import {fetchOfferInfoByTokenAction} from 'reducers/depreacted/hotels/bookAndPayPage/offerInfoByToken/actions';
import {
    resetOrderInfoAction,
    getOrderInfoActions,
    createOrderActions,
    stopCreateOrderActions,
} from 'reducers/depreacted/hotels/bookAndPayPage/orderInfo/actions';
import {IBookPromoCodesReducer} from 'reducers/depreacted/hotels/bookAndPayPage/promoCodesInfo/reducer';
import {resetBookUserSelectAction} from 'reducers/depreacted/hotels/bookAndPayPage/bookUserSelectOptions/actions';

import {getBookPromoCodes} from 'selectors/depreacted/hotels/book/getBookPromoCodes';

import {reachGoal} from 'utilities/metrika';
import {getCSRFTokenHeaderGenerator} from 'utilities/csrfToken/getCSRFTokenHeaderGenerator';
import {createNormalizedGuestsFormCopy} from '../utilities/createNormalizedGuestsFormCopy';
import {getPaymentTestContextToken} from 'utilities/url/getPaymentTestContextToken';

import {hotelBookService} from 'serviceProvider';

const prepareErrorResponse = (
    errorResponse?: TCreateOrderErrorResponse,
): IOrderError => {
    return {
        code: errorResponse?.data?.error?.code ?? UNKNOWN_ORDER_ERROR,
        offerInfo: errorResponse?.data?.error?.offerInfo,
    };
};

function preparePromoCodes(promoCodes?: IAppliedPromoCodesInfo) {
    return promoCodes?.codeApplicationResults
        .filter(
            promoCode => promoCode.type === EPromoCodesApplicationType.SUCCESS,
        )
        .map(promoCode => promoCode.code);
}

const createHotelOrder = function* (
    action: ReturnType<typeof createOrderActions.request>,
): SagaIterator {
    const {payload: actionPayload} = action;
    const {deviceType, finishPaymentPagePath, ...payload} = actionPayload;

    payload.formValues = createNormalizedGuestsFormCopy(payload.formValues);

    try {
        const {data}: IBookPromoCodesReducer = yield select(getBookPromoCodes);

        const requestPayload: ICreateOrderRequest = {
            ...payload,
            promoCodes: preparePromoCodes(data),
            paymentTestContextToken: getPaymentTestContextToken(),
        };

        /* Create order request */
        const csrfTokenHeader = yield call(getCSRFTokenHeaderGenerator);
        const {data: orderRequestResponse}: TCreateOrderResponse = yield call(
            hotelBookService.provider().createOrder,
            requestPayload,
            csrfTokenHeader,
        );

        const {orderId} = orderRequestResponse;

        reachGoal(EHotelsGoal.BOOK_CREATE_ORDER_SUCCESS);

        /* Start polling order action */
        yield put(
            getOrderInfoActions.request({
                orderId,
                finishPaymentPagePath,
            }),
        );

        /* Reset user payment info form */
        yield put(resetBookUserSelectAction());
    } catch (e) {
        const error = prepareErrorResponse(e.response);
        const {code} = error;

        if (code === UNKNOWN_OFFER_ERROR) {
            reachGoal(EHotelsGoal.BOOK_CREATE_ORDER_ERROR, {
                hotels: {orderError: code.toLowerCase()},
            });

            /* Create order finished with error */
            yield put(createOrderActions.failure());
        } else {
            reachGoal(EHotelsGoal.BOOK_CREATE_ORDER_ERROR, {
                hotels: {orderError: code && code.toLowerCase()},
            });

            yield put(
                batchActions([
                    /* Fetch offer info finished with error */
                    fetchOfferInfoByTokenAction.failure({error}),
                    /* Reset order */
                    resetOrderInfoAction(),
                ]),
            );
        }
    }
};

function* createHotelOrderOrStop(
    action: ReturnType<typeof createOrderActions.request>,
) {
    yield race([
        call(createHotelOrder, action),
        take(getType(stopCreateOrderActions)),
    ]);
}

export default function* (): SagaIterator {
    /* Create order watcher */
    yield takeEvery(
        getType(createOrderActions.request),
        createHotelOrderOrStop,
    );
}
