import {takeLatest, call, take, race, put, delay} from 'redux-saga/effects';
import _get from 'lodash/get';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {
    RESERVED,
    RESERVED_WITH_RESTRICTIONS,
    IN_PROGRESS,
    AWAITS_PAYMENT,
    CANCELLED,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';

/* Redux and Types */
import {
    fetchOfferInfoByTokenAction,
    updateOfferInfoAction,
} from 'reducers/depreacted/hotels/bookAndPayPage/offerInfoByToken/actions';
import {
    getOrderInfoActions,
    stopOrderInfoPollingAction,
    orderNeedAuthorizationAction,
    startOrderPaymentActions,
} from 'reducers/depreacted/hotels/bookAndPayPage/orderInfo/actions';
import {setBookUserSelectAction} from 'reducers/depreacted/hotels/bookAndPayPage/bookUserSelectOptions/actions';
import {setRefundModalVisible} from 'reducers/depreacted/hotels/bookAndPayPage/orderCancelAndRefundableInfo/actions';

import {getReason} from 'selectors/depreacted/hotels/book/orderCancellationDetails/getReason';

import {hotelBookService} from 'serviceProvider';

/* Constants */
const POLLING_ORDER_DELAY = 1000;

const setBookUserSelect = function* (guestsInfo: any) {
    const {selectedBedGroupIndex} = guestsInfo;

    if (selectedBedGroupIndex) {
        yield put(
            setBookUserSelectAction({
                bedsGroupId: String(selectedBedGroupIndex),
            }),
        );
    }
};

const startPollingOrder = function* (
    action: ReturnType<typeof getOrderInfoActions.request>,
) {
    const {payload} = action;
    let prevStatus: string | null = null;

    while (true) {
        try {
            const {withRestrictions, isPaymentStarted} = payload;

            /* Call getOrderStatus request */
            const orderStatus: any = yield call(
                hotelBookService.provider().getOrderStatus,
                {orderId: payload.orderId},
            );

            if (
                orderStatus.status === IN_PROGRESS ||
                orderStatus.status === prevStatus
            ) {
                prevStatus = orderStatus.status;

                /* Continue polling after delay */
                yield delay(POLLING_ORDER_DELAY);

                continue;
            }

            /* Create getOrder request */
            const order: any = yield call(
                hotelBookService.provider().getOrder,
                {orderId: payload.orderId},
            );

            const {
                status,
                orderId,
                orderCancellationDetails,
                offerInfo,
                guestsInfo,
            } = order;

            if (status === IN_PROGRESS) {
                /* Continue polling after delay */
                yield delay(POLLING_ORDER_DELAY);

                continue;
            }

            /* Update offer info */
            yield put(updateOfferInfoAction({data: offerInfo}));

            /* Update polling status and continue */
            yield put(getOrderInfoActions.success({data: order}));

            if (
                status === RESERVED ||
                (status === RESERVED_WITH_RESTRICTIONS && withRestrictions)
            ) {
                if (isPaymentStarted) {
                    prevStatus = status;

                    /* Continue polling after delay */
                    yield delay(POLLING_ORDER_DELAY);

                    continue;
                }

                const {finishPaymentPagePath} = payload;

                /* Finish polling and start payment after reserve offer */
                yield put(
                    startOrderPaymentActions.request({
                        orderId,
                        finishPaymentPagePath,
                    }),
                );
            }

            const reason = getReason(orderCancellationDetails);

            if (
                status === CANCELLED &&
                reason === 'OrderCancellationDetailsReasonType.INVALID_INPUT'
            ) {
                const {
                    offerOrderInfo: {token, label},
                } = offerInfo;

                /* Finish polling and start fetch offer info */
                yield put(fetchOfferInfoByTokenAction.request({token, label}));

                /* Set book user select after payment failed */
                yield setBookUserSelect(guestsInfo);

                yield put(setRefundModalVisible(false));
            }

            prevStatus = status;

            if (status === AWAITS_PAYMENT) {
                yield delay(POLLING_ORDER_DELAY);

                continue;
            }

            /* Finish polling action */
            yield put(stopOrderInfoPollingAction());
        } catch (e) {
            const responseStatusCode = _get(e, 'response.status');

            if (
                responseStatusCode ===
                'StatusCodes.NEED_AUTHORIZATION_STATUS_CODE'
            ) {
                /* Finish polling by NEED_AUTHORIZATION_STATUS_CODE */
                yield put(orderNeedAuthorizationAction());
            } else {
                /* Finish polling with error status */
                yield put(getOrderInfoActions.failure());
            }
        }
    }
};

const watchPollingOrder = function* (
    action: ReturnType<typeof getOrderInfoActions.request>,
) {
    yield race([
        call(startPollingOrder, action),
        take(getType(stopOrderInfoPollingAction)),
        take(getType(getOrderInfoActions.failure)),
        take(getType(orderNeedAuthorizationAction)),
    ]);
};

export default function* (): SagaIterator {
    /* Start polling watcher */
    yield takeLatest(getType(getOrderInfoActions.request), watchPollingOrder);
}
