import {
    takeLatest,
    put,
    all,
    call,
    delay,
    select,
    CallEffect,
    race,
    take,
} from 'redux-saga/effects';
import {SagaIterator} from 'redux-saga';

import {
    IHotelInfo,
    IHotelInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/IHotelInfo';

import {HOTEL_INFO_ACTION_TYPES} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/actionTypes';
import {IHotelInfoReducer} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/reducer';
import {
    getHotelInfoActions,
    getHotelInfoOffersActions,
    getSimilarHotelsActions,
    getAdditionalHotelInfoAction,
    syncHotelInfoSearchParamsWithLocation,
} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/actions';
import {resetHotelImagesAction} from 'reducers/depreacted/hotels/hotelPage/hotelImages/actions';
import {
    getHotelReviewsActions,
    resetHotelReviewsAction,
} from 'reducers/depreacted/hotels/hotelPage/reviews/list/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {getHotelInfo} from 'selectors/depreacted/hotels/hotel/mainTab/getHotelInfo';

import sendMetrikaExtraVisitAndUserParams from 'projects/depreacted/hotels/utilities/metrika/sendMetrikaExtraVisitAndUserParams';

import {hotelSearchService} from 'serviceProvider';

/* Constants */

const DELAY_AFTER_FETCH_HOTEL_INFO = 200;

export const fetchHotelInfoOffersAndSimilarHotelsIfNeed = function* () {
    const reduxState: StoreInterface = yield select();
    const {data} = getHotelInfo(reduxState);
    const canStartHotelInfoOffersPolling =
        !data?.offersInfo?.offerSearchProgress?.finished;
    const canStartSimilarHotelsInfoPolling =
        !data?.similarHotelsInfo?.offerSearchProgress.finished;

    if (canStartHotelInfoOffersPolling) {
        yield put(getHotelInfoOffersActions.request());
    }

    if (canStartSimilarHotelsInfoPolling) {
        yield put(getSimilarHotelsActions.request());
    }
};

/* AdditionHotelInfo */
const watchStartFetchAdditionHotelInfo = function* () {
    /* Offers */
    yield fetchHotelInfoOffersAndSimilarHotelsIfNeed();

    /* Location */
    yield put(syncHotelInfoSearchParamsWithLocation());
};

/* HotelInfo */
const startHotelInfoRequest = function* (
    requestParams: IHotelInfoRequestParams,
): Generator<
    CallEffect<IHotelInfo>,
    IHotelInfo | undefined,
    IHotelInfoReducer
> {
    const {data} = yield call(
        hotelSearchService.provider().getHotelInfo,
        requestParams,
    );

    return data;
};

const startFetchHotelInfo = function* (
    payload: IHotelInfoRequestParams,
): SagaIterator {
    try {
        yield put(resetHotelImagesAction());

        const hotelInfo = yield call(startHotelInfoRequest, payload);

        sendMetrikaExtraVisitAndUserParams(hotelInfo.extraVisitAndUserParams);

        yield put(getHotelInfoActions.success(hotelInfo));
        yield put(resetHotelReviewsAction());
        yield put(getHotelReviewsActions.success(hotelInfo.reviewsInfo));

        yield delay(DELAY_AFTER_FETCH_HOTEL_INFO);
        yield put(getAdditionalHotelInfoAction());
    } catch {
        yield put(getHotelInfoActions.failure());
    }
};

const watchStartFetchHotelInfo = function* (
    action: ReturnType<typeof getHotelInfoActions.request>,
) {
    const {payload} = action;

    yield race({
        startAction: call(startFetchHotelInfo, payload),
        stopAction: take(HOTEL_INFO_ACTION_TYPES.STOP_HOTEL_INFO_ACTIONS),
    });
};

/* Total sagas */
export default function* (): SagaIterator {
    yield all([
        yield takeLatest(
            HOTEL_INFO_ACTION_TYPES.START_HOTEL_INFO_REQUEST,
            watchStartFetchHotelInfo,
        ),
        yield takeLatest(
            HOTEL_INFO_ACTION_TYPES.GET_ADDITIONAL_HOTEL_INFO,
            watchStartFetchAdditionHotelInfo,
        ),
    ]);
}
