import {
    takeLatest,
    takeEvery,
    put,
    all,
    call,
    select,
} from 'redux-saga/effects';
import {batchActions} from 'redux-batched-actions';
import {SagaIterator} from 'redux-saga';

import {HOTEL_REVIEWS_ACTION_TYPES} from 'reducers/depreacted/hotels/hotelPage/reviews/list/actionTypes';
import {HOTEL_ADD_REVIEW_ACTION_TYPES} from 'reducers/depreacted/hotels/hotelPage/reviews/add/actionTypes';
import {HOTEL_EDIT_REVIEW_ACTION_TYPES} from 'reducers/depreacted/hotels/hotelPage/reviews/edit/actionTypes';
import {
    getHotelReviewsActions,
    setActiveKeywordAction,
    setHotelReviewInfoReactionAction,
    setHotelReviewReactionRequestActions,
    resetHotelReviewsListAction,
    setHotelReviewReactionAction,
    filterHotelReviewsActions,
    sortingHotelReviewsAction,
    setHotelReviewsSortingOptionAction,
} from 'reducers/depreacted/hotels/hotelPage/reviews/list/actions';
import {addHotelReviewActions} from 'reducers/depreacted/hotels/hotelPage/reviews/add/actions';
import {editHotelReviewActions} from 'reducers/depreacted/hotels/hotelPage/reviews/edit/actions';
import {
    setHotelInfoReviewReactionAction,
    setHotelInfoTotalReviewsAction,
} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/actions';
import {HOTEL_DELETE_REVIEW_ACTION_TYPES} from 'reducers/depreacted/hotels/hotelPage/reviews/delete/actionTypes';
import {deleteHotelReviewActions} from 'reducers/depreacted/hotels/hotelPage/reviews/delete/actions';

import {
    getHotelConfirmedReactions,
    GetHotelConfirmedReactionsType,
} from 'selectors/depreacted/hotels/hotel/reviewsTab/getHotelConfirmedReactions';

import {getSearchParamsByReduxHotelState} from 'projects/depreacted/hotels/utilities/getHotelPageParams/getSearchParamsByReduxHotelState';

import {hotelSearchService} from 'serviceProvider';

/* Request */
const getHotelReviewsRequest = function* (quantitativeAndFilterParams: any) {
    const {hotelSlug, parentRequestId} =
        yield getSearchParamsByReduxHotelState();

    const {data} = yield call(hotelSearchService.provider().getHotelReviews, {
        hotelSlug,
        parentRequestId,
        ...quantitativeAndFilterParams,
    });

    return data;
};

const sendHotelReviewReactionRequest = function* (requestParams: any) {
    yield call(
        hotelSearchService.provider().setHotelReviewReaction,
        requestParams,
    );
};

/* Sagas */
const getHotelReviews = function* (
    action: ReturnType<typeof getHotelReviewsActions.request>,
): SagaIterator {
    const {payload} = action;

    try {
        const hotelReviews = yield call(getHotelReviewsRequest, payload);

        yield put(
            batchActions([
                getHotelReviewsActions.success(hotelReviews),
                setHotelInfoTotalReviewsAction(hotelReviews),
            ]),
        );
    } catch {
        yield put(getHotelReviewsActions.failure());
    }
};

const filterHotelReviews = function* (
    action: ReturnType<typeof filterHotelReviewsActions>,
): SagaIterator {
    const {payload} = action;

    yield put(setActiveKeywordAction(payload.keyPhraseFilter));
    yield put(resetHotelReviewsListAction());

    try {
        const hotelReviews = yield call(getHotelReviewsRequest, payload);

        yield put(getHotelReviewsActions.success(hotelReviews));
    } catch {
        yield put(getHotelReviewsActions.failure());
    }
};

const sortingHotelReviews = function* (
    action: ReturnType<typeof sortingHotelReviewsAction>,
): SagaIterator {
    const {payload} = action;
    const currentSortingOption =
        payload.textReviewRanking || 'ETextReviewRankingType.RELEVANCE_ORG';

    yield put(resetHotelReviewsListAction());
    yield put(setHotelReviewsSortingOptionAction(currentSortingOption));

    try {
        const hotelReviews = yield call(getHotelReviewsRequest, payload);

        yield put(getHotelReviewsActions.success(hotelReviews));
    } catch {
        yield put(getHotelReviewsActions.failure());
    }
};

/* Оптимистичная установка */
const setHotelReviewReaction = function* (
    action: ReturnType<typeof setHotelReviewReactionRequestActions.request>,
) {
    const {payload} = action;

    yield put(setHotelReviewReactionAction(payload));
    yield put(setHotelInfoReviewReactionAction(payload));
};

/* Запрос на установку реакции */
const setHotelReviewReactionRequest = function* (
    action: ReturnType<typeof setHotelReviewReactionRequestActions.request>,
) {
    const {
        payload: {previousUserReaction, reviewId, userReaction},
    } = action;

    const confirmedReactions: ReturnType<GetHotelConfirmedReactionsType> =
        yield select(getHotelConfirmedReactions);
    const confirmedReaction = confirmedReactions[reviewId];
    const reaction = confirmedReaction
        ? confirmedReaction
        : previousUserReaction;

    try {
        const {hotelSlug} = yield getSearchParamsByReduxHotelState();
        const reviewReactionRequestParams = {
            hotelSlug,
            reviewId,
            userReaction,
        };

        yield sendHotelReviewReactionRequest(reviewReactionRequestParams);

        yield put(
            setHotelReviewReactionRequestActions.success(
                reviewReactionRequestParams,
            ),
        );
    } catch {
        yield put(
            setHotelReviewInfoReactionAction({
                reviewId,
                userReaction: reaction,
            }),
        );
    }
};

const addHotelReview = function* (
    action: ReturnType<typeof addHotelReviewActions.request>,
): SagaIterator {
    const {payload} = action;
    const {hotelSlug} = yield call(getSearchParamsByReduxHotelState);

    try {
        const response = yield call(
            hotelSearchService.provider().addHotelReview,
            {
                ...payload,
                hotelSlug,
            },
        );

        yield put(addHotelReviewActions.success(response.data));
    } catch {
        yield put(addHotelReviewActions.failure());
    }
};

const editHotelReview = function* (
    action: ReturnType<typeof editHotelReviewActions.request>,
): SagaIterator {
    const {payload} = action;
    const {hotelSlug} = yield call(getSearchParamsByReduxHotelState);

    try {
        const response = yield call(
            hotelSearchService.provider().editHotelReview,
            {
                ...payload,
                hotelSlug,
            },
        );

        yield put(editHotelReviewActions.success(response.data));
    } catch {
        yield put(editHotelReviewActions.failure());
    }
};

const deleteHotelReview = function* (
    action: ReturnType<typeof deleteHotelReviewActions.request>,
): SagaIterator {
    const {payload} = action;
    const {hotelSlug} = yield call(getSearchParamsByReduxHotelState);

    try {
        const response = yield call(
            hotelSearchService.provider().deleteHotelReview,
            {...payload, hotelSlug},
        );

        yield put(deleteHotelReviewActions.success(response.data));
    } catch {
        yield put(deleteHotelReviewActions.failure());
    }
};

export default function* (): SagaIterator {
    yield all([
        yield takeLatest(
            HOTEL_REVIEWS_ACTION_TYPES.START_HOTEL_REVIEWS_REQUEST,
            getHotelReviews,
        ),
        yield takeLatest(
            HOTEL_REVIEWS_ACTION_TYPES.FILTER_HOTEL_REVIEWS_REQUEST,
            filterHotelReviews,
        ),
        yield takeEvery(
            HOTEL_REVIEWS_ACTION_TYPES.SET_HOTEL_REVIEW_INFO_REACTION,
            setHotelReviewReaction,
        ),
        yield takeEvery(
            HOTEL_REVIEWS_ACTION_TYPES.START_HOTEL_REVIEW_REACTION_REQUEST,
            setHotelReviewReactionRequest,
        ),
        yield takeEvery(
            HOTEL_REVIEWS_ACTION_TYPES.SORTING_HOTEL_REVIEWS_LIST,
            sortingHotelReviews,
        ),
        yield takeEvery(
            HOTEL_ADD_REVIEW_ACTION_TYPES.ADD_HOTEL_REVIEW_REQUEST,
            addHotelReview,
        ),
        yield takeEvery(
            HOTEL_EDIT_REVIEW_ACTION_TYPES.EDIT_HOTEL_REVIEW_REQUEST,
            editHotelReview,
        ),
        yield takeEvery(
            HOTEL_DELETE_REVIEW_ACTION_TYPES.DELETE_HOTEL_REVIEW_REQUEST,
            deleteHotelReview,
        ),
    ]);
}
