import {takeLatest, call, race, put, select} from 'redux-saga/effects';
import {SagaIterator} from '@redux-saga/core';

import {
    ICountHotelsRequestParams,
    ICountHotelsResponse,
} from 'server/api/HotelsSearchAPI/types/ICountHotels';

import {FILTERS_ACTION_TYPES} from 'reducers/depreacted/hotels/searchPage/filters/actionTypes';
import {updateCountFiltersActions} from 'reducers/depreacted/hotels/searchPage/filters/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {getMap, GetMapType} from 'selectors/depreacted/hotels/search/map/getMap';
import {
    getHotelsFilters,
    GetFiltersType,
} from 'selectors/depreacted/hotels/search/filters/hotelsFiltersSelector';
import {getSortInfo} from 'selectors/depreacted/hotels/search/sort/getSortInfo';

import {getFilterParams} from 'sagas/depreacted/hotels/utilities/getSearchPageFilterParams';
import {getOfferRequestParams} from 'sagas/depreacted/hotels/utilities/getSearchPageOfferRequestParams';

import {convertBoundsToString} from 'components/YandexMaps/utilities';
import {getAttributionParams} from 'projects/depreacted/hotels/utilities/getAttributionParams/getAttributionParams';

import {hotelsSearchService} from 'serviceProvider';

/* Generator Helpers */
const getSearchParamsByState =
    function* (): SagaIterator<ICountHotelsRequestParams> {
        const reduxState: StoreInterface = yield select();
        const {mapBoundsBySearch}: ReturnType<GetMapType> = getMap(reduxState);
        const {currentFilters}: ReturnType<GetFiltersType> =
            getHotelsFilters(reduxState);

        /* OfferRequestParam */
        const {adults, childrenAges, checkoutDate, checkinDate} =
            getOfferRequestParams(reduxState);

        /* Filters */
        const {filterAtoms, filterPriceTo, filterPriceFrom} =
            getFilterParams(currentFilters);

        /* Map */
        const bbox = mapBoundsBySearch
            ? convertBoundsToString(mapBoundsBySearch)
            : undefined;

        return {
            bbox,
            adults,
            childrenAges,
            checkoutDate,
            checkinDate,
            filterAtoms,
            filterPriceTo,
            filterPriceFrom,
        };
    };

const startUpdateCountFilters = function* (): SagaIterator {
    const countHotelsRequest = hotelsSearchService.provider().countHotels;

    try {
        const searchParamsByState = yield call(getSearchParamsByState);
        const {sortOrigin, sortInfo} = yield select(getSortInfo);

        const debugAndMetaAttributionSearchParams = getAttributionParams();

        const {data: countHotelsResponse}: {data: ICountHotelsResponse} =
            yield call(countHotelsRequest, {
                sortOrigin: sortOrigin || sortInfo?.sortOrigin,
                ...searchParamsByState,
                ...debugAndMetaAttributionSearchParams,
            });

        yield put(updateCountFiltersActions.success(countHotelsResponse));
    } catch (e) {
        yield put(updateCountFiltersActions.failure());
    }
};

const watchUpdateCountFilters = function* () {
    yield race({
        startAction: call(startUpdateCountFilters),
    });
};

export default function* (): SagaIterator {
    yield takeLatest(
        FILTERS_ACTION_TYPES.START_UPDATE_COUNT_FILTERS,
        watchUpdateCountFilters,
    );
}
