import {createSelector} from 'reselect';
import keyBy from 'lodash/keyBy';

import sortCountries from 'reducers/common/countries/utilities/sortCountries';
import {CountriesStateType} from 'reducers/common/countries/reducer';
import {StoreInterface} from 'reducers/storeTypes';

const getCountriesState = (state: StoreInterface): CountriesStateType =>
    state.common.countries;

export const getAllCountries = createSelector(
    getCountriesState,
    ({countries, countryData}) =>
        sortCountries(countries.map(geoId => countryData[geoId])),
);

export const getGeoIdToCode2Converter = createSelector(
    getCountriesState,
    ({countryData}) =>
        (geoId: number): string =>
            (countryData[geoId] && countryData[geoId].code2) || '',
);

export const getCode2ToGeoIdConverter = createSelector(
    getAllCountries,
    countries => {
        const code2Map = keyBy(countries, 'code2');

        return (code2: string): number | undefined =>
            (code2Map[code2] && code2Map[code2].geoId) || undefined;
    },
);

export const getCodesAndTitles = createSelector(getAllCountries, countries =>
    countries.map<{value: string; data: string}>(({code2, title}) => ({
        value: code2!,
        data: title,
    })),
);

export const getAllCountriesData = createSelector(
    getCountriesState,
    ({countryData}) => countryData,
);

export const getCountriesReducerState = createSelector(
    getCountriesState,
    ({isLoading, isFetched, isError}) => ({
        isFetched,
        isLoading,
        isError,
    }),
);
