// -------------------------------------------------
// Validation
// -------------------------------------------------
import {IFieldValidation, TFieldGroupId, IFormValues} from './form';

export enum EValidationType {
    required = 'required',
    available = 'available',

    regex = 'regex',
    maxLength = 'maxLength',
    minLength = 'minLength',

    maxValue = 'maxValue',
    minValue = 'minValue',

    oneOf = 'oneOf',
    notOneOf = 'notOneOf',

    validDate = 'validDate',
    maxDate = 'maxDate',
    minDate = 'minDate',
    maxDateFromToday = 'maxDateFromToday',
    minDateFromToday = 'minDateFromToday',
    maxDateFromField = 'maxDateFromField',
    minDateFromField = 'minDateFromField',
}

type TOneOfVales = (string | boolean | number | null | undefined)[];

type TDateOffsetScale = 'year' | 'month' | 'day';

export interface IDateOffset {
    offset: number;
    scale: TDateOffsetScale;
    strict?: boolean;
    comparisonScale?: TDateOffsetScale;
}

export interface IDateOffsetFromField extends IDateOffset {
    field: IFieldPath;
}

export interface IRegexInfo {
    regex: string;
    flags?: string;
}

export type TValidationParams =
    | number
    | string
    | boolean
    | IRegexInfo
    | TOneOfVales
    | IDateOffset
    | IDateOffsetFromField;

/** Общий интерфейс для наследования типов валидации */
interface IBaseValue<T extends EValidationType, P extends TValidationParams> {
    type: T;
    params: P;
}

interface IWithError {
    errorMessage: string;
}

/* Common validations */

interface IRequiredValidation
    extends IBaseValue<EValidationType.required, boolean> {}
interface IAvailableValidation
    extends IBaseValue<EValidationType.available, boolean> {}

export type TCommonValue = IRequiredValidation | IAvailableValidation;
export type TCommonValidation = TCommonValue & IWithError;

/* String validations */

interface IRegexValidation
    extends IBaseValue<EValidationType.regex, string | IRegexInfo> {}
interface IMaxLengthValidation
    extends IBaseValue<EValidationType.maxLength, number> {}
interface IMinLengthValidation
    extends IBaseValue<EValidationType.minLength, number> {}

export type TStringValue =
    | TCommonValue
    | IRegexValidation
    | IMaxLengthValidation
    | IMinLengthValidation;

export type TStringValidation = TStringValue & IWithError;

/* Number validations */

interface IMaxValueValidation
    extends IBaseValue<EValidationType.maxValue, number> {}
interface IMinValueValidation
    extends IBaseValue<EValidationType.minValue, number> {}

export type TNumberValue =
    | TCommonValidation
    | IMaxValueValidation
    | IMinValueValidation;

export type TNumberValidation = TNumberValue & IWithError;

/* Select validations */

interface IOneOfValuesValidation
    extends IBaseValue<EValidationType.oneOf, TOneOfVales> {}
interface INotOneOfValuesValidation
    extends IBaseValue<EValidationType.notOneOf, TOneOfVales> {}

export type TSelectValue =
    | TCommonValidation
    | IOneOfValuesValidation
    | INotOneOfValuesValidation;
export type TSelectValidation = TSelectValue & IWithError;

/* Date validations */

/** Строка в стандартизированном формате, например '22.01.2010' */
interface IValidDateValidation
    extends IBaseValue<EValidationType.validDate, string> {}

interface IMaxDateValidation
    extends IBaseValue<EValidationType.maxDate, string> {}
interface IMinDateValidation
    extends IBaseValue<EValidationType.minDate, string> {}

interface IMaxDateFromTodayValidation
    extends IBaseValue<EValidationType.maxDateFromToday, IDateOffset> {}
interface IMinDateFromTodayValidation
    extends IBaseValue<EValidationType.minDateFromToday, IDateOffset> {}

interface IMaxDateFromFieldValidation
    extends IBaseValue<
        EValidationType.maxDateFromField,
        IDateOffsetFromField
    > {}
interface IMinDateFromFieldValidation
    extends IBaseValue<
        EValidationType.minDateFromField,
        IDateOffsetFromField
    > {}

export type TDateValue =
    | TCommonValidation
    | IValidDateValidation
    | IMaxDateValidation
    | IMinDateValidation
    | IMaxDateFromTodayValidation
    | IMinDateFromTodayValidation
    | IMaxDateFromFieldValidation
    | IMinDateFromFieldValidation;

export type TDateValidation = TDateValue & IWithError;

/** Main validation type */
export type TValidations =
    | TStringValidation[]
    | TNumberValidation[]
    | TDateValidation[]
    | TSelectValidation[];

// -------------------------------------------------
// Dependent validation
// -------------------------------------------------

export type TDependentValue =
    | TStringValue[]
    | TNumberValue[]
    | TDateValue[]
    | TSelectValue[];

export enum EDependentConditionType {
    EVERY = 'every',
    SOME = 'some',
}

interface IFieldPath {
    /**
     * Id группы значений в форме.
     * Например: документ пассажира при брониваронии, контактная информация при бронированиии.
     *
     * Если Id не указан, то проверка условий происходит по текущей группе.
     */
    fieldGroupId?: TFieldGroupId;
    fieldName: string;

    /**
     * Когда есть зависимость от массива сущностей
     * Например: Хотя бы у одного пассажира документ - Паспорт РФ
     * Например: Все пассажиры дети
     */
    type?: EDependentConditionType;
}

export interface IDependencyCondition {
    path: IFieldPath;
    value: TDependentValue;
}

/**
 * Валидация соотвествующая комбинации значений полей.
 *
 * При созданни пассажира в аккаунте, телефон будет обязательными, если:
 * 1) поставить галку  о согласии на рассылку (agreement.notification.value === true),
 * 2) при пустом email (contacts.email.value === '')
 */
export interface IDependentValidation {
    conditions: IDependencyCondition[];
    validation: IFieldValidation<TValidations>;
}

export type TDependentValidations = IDependentValidation[];

export interface IValidationOptions {
    isSubmit?: boolean;
    isStrictStructure?: boolean;
}

export interface IFormValidationData {
    formValues: IFormValues;
    /** Путь до группы (объекта) в котором располагается значение */
    groupPath: (string | number)[];
}
