import {IResetFilterInfo} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import IPrice from 'utilities/currency/PriceInterface';

/* Active Filters Params */
export interface IFilterParams {
    filterAtoms?: FilterAtomType[];
    filterPriceFrom?: number;
    filterPriceTo?: number;
}

/* Atom */
export type FilterAtomType = string;

/* Base Filter */
export interface IBasicFilter {
    name: string;
    id: string;
    effect?: string;
    enabled: boolean;
    hint: string;
    atoms: FilterAtomType[];
}

export enum EQuickControlType {
    QUICK_FILTER = 'quick-filter',
    QUICK_FILTER_PRICE = 'quick-price-filter',
    QUICK_SORT = 'quick-sort',
}
export interface IQuickControl {
    // Уникальный идентификатор, нужен для оптимизации верстки
    id: string;
    type: EQuickControlType;
    // Эффект, как отображать этот фильтр.
    // Примеры: подкраска рейтинга, спец-значок для "только спецпредложения".
    // Примеры: для быстрого фильтра -- какой-нибудь спец-цвет или другая штука.
    effect?: EQuickFilterEffect;
    // Если enabled == true, то можно на контрол нажимать.
    // Если enabled == false, то нажимать нельзя (будет пустая выдача).
    enabled: boolean;
}

// Быстрый фильтр
export type TQuickFilter = IQuickControl & {
    // Название.
    name: string;
    // всплывающая подсказка. Если отсутствует - то её нет
    hint?: string;

    // Для определения факта нажатости быстрофильтра:
    // если в текущем стейте фильтра все атомы из atomsOn есть, и нет ни одного из atomsOff - то быстрофильтр нажат
    // При нажатии на быстрофильтр: все атомы из atomsOff - удаляем из стейта, Все атомы из atomsOn - добавляем в стейт.
    // При отжатии быстрофильтра: Все атомы из atomsOn удаляем из стейта. atomsOff в стейте быть не могло, иначе фильтр бы не был нажат
    atomsOn: FilterAtomType[];
    atomsOff: FilterAtomType[];

    type: EQuickControlType.QUICK_FILTER;
};

export type TQuickPriceFilter = IQuickControl & {
    // Название
    name: string;
    type: EQuickControlType.QUICK_FILTER_PRICE;
};

export type TQuickSort = IQuickControl & {
    type: EQuickControlType.QUICK_SORT;
};

type TQuickControlTypes = TQuickPriceFilter | TQuickSort | TQuickFilter;

export enum EQuickFilterEffect {
    YANDEX_PLUS = 'yandex-plus',
    MIR_CASHBACK = 'mir-cashback',
}

/* Filter Group */
export interface IBasicFilterGroup {
    name: string;
    id: string;
    type: 'SINGLE' | 'MULTI';
    items: IBasicFilter[];
}

export enum EDetailedFiltersBatchItemType {
    /**
     * фильтр по цене, надо взять его из IFilterInfo
     */
    PRICE = 'PRICE',
    /**
     * обычная группа фильтров (передаётся в detailedFilters)
     */
    GROUP = 'GROUP',
}

export interface IDetailedFilterItem {
    /**
     * Что должно быть в этом элементе
     */
    type: EDetailedFiltersBatchItemType;

    // Заполняется, если type == 'GROUP'
    detailedFilters?: IBasicFilterGroup;
}

export interface IDetailedFiltersBatch {
    items: IDetailedFilterItem[];
}

/* Price Filter */
export interface IRangePriceFilter extends IPriceFilter {
    minValue?: number;
    maxValue?: number;
}

export interface IRangePriceFilterEstimateValues {
    minPriceEstimate: IPrice['value'];
    maxPriceEstimate: IPrice['value'];
}

export interface IPriceFilter extends IRangePriceFilterEstimateValues {
    currency: IPrice['currency'];
    histogramBounds?: number[];
    histogramCounts?: number[];
}

/* Change Filter Payload */
export interface IChangeFiltersTargetType {
    targetFiltersType: 'CURRENT' | 'ALL';
}

export interface IChangeFilterGroupPayload {
    atomsOn?: FilterAtomType[];
    atomsOff?: FilterAtomType[];
}

export interface IChangeFilterGroupMeta {
    filterId: string;
    itemId: string;
}

export interface IChangeFilterGroupPayloadWithTarget
    extends IChangeFilterGroupPayload,
        IChangeFiltersTargetType {}

export interface IChangePriceFilterPayload {
    minValue: number;
    maxValue: number;
}

export interface IChangePriceFilterPayloadWithTarget
    extends IChangePriceFilterPayload,
        IChangeFiltersTargetType {}

export interface IChangeTextFilterPayloadWithTarget
    extends IChangeFiltersTargetType {}

/* Filters Info */

export type ActiveFilterAtomsType = Record<FilterAtomType, boolean>;

export type TCollapsedFilters = PartialRecord<string, true>;

export interface IFiltersInfo {
    params: IFilterParams;
    quickFilters: TQuickFilter[];
    detailedFilters?: IDetailedFilterItem[];
    detailedFiltersBatches: IDetailedFiltersBatch[];
    priceFilter: IPriceFilter;
    resetFilterInfo: IResetFilterInfo | null; // Информация для сброса фильтров. Если null, то блок сброса фильтров не рисуем
}

export interface ISearchControlsInfo {
    quickControls: TQuickControlTypes[];
}

export interface IFullFilters {
    foundHotelCount?: number;
    totalHotelCount?: number;
    quickFilters?: TQuickFilter[];
    quickControls?: TQuickControlTypes[];
    detailedFilters?: IDetailedFilterItem[];
    detailedFiltersBatches?: IDetailedFiltersBatch[];
    priceFilter?: IRangePriceFilter;
    activeFilterAtoms: ActiveFilterAtomsType;
}

export interface IFullFiltersInfo {
    needSyncPermanentFiltersWithServer: boolean;
    needSyncCurrentFiltersWithServer: boolean;
    permanentFilters: IFullFilters;
    currentFilters: IFullFilters;
    totalActiveFilters: number;
    isLeftFiltersEnabled: boolean;
    collapsedFilters: TCollapsedFilters;
}
