import moment from 'moment';

import {IHumanizePeriod} from 'types/common/date/IHumanizePeriod';

import {DateLikeType} from 'utilities/dateUtils/index';
import {
    HUMAN,
    HUMAN_SHORT,
    HUMAN_SHORT_WITH_SHORT_YEAR,
    HUMAN_WITH_YEAR,
} from 'utilities/dateUtils/formats';
import {CHAR_MIDDLE_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';

/**
 * Выводит самый человекочитаемый период дат:
 * если даты совпадают, то выводит только одну дату;
 * если даты в одном месяце, то имя месяца группируется;
 * если дата начала из года отличного от текущего, то указывается год;
 * поддерживается сокращенный формат
 * @param start
 * @param end
 * @param options
 */
export default function humanizePeriod(
    start: DateLikeType,
    end: DateLikeType,
    options: IHumanizePeriod = {},
): string {
    const {
        checkSameDate = true,
        short = false,
        checkAnotherEndYear = false,
    } = options;

    const startMoment = moment(start);
    const endMoment = moment(end);

    const isAnotherStartYear = startMoment.isAfter(moment(), 'year');

    if (checkSameDate && startMoment.isSame(endMoment, 'day')) {
        const format = getFormatWithNbsp(getFormatForDate(isAnotherStartYear));

        return startMoment.format(format);
    }

    const stateDateFormat = getFormatWithNbsp(
        getFormatForInterval(isAnotherStartYear, short),
    );

    const isAnotherEndYear = checkAnotherEndYear
        ? endMoment.isAfter(moment(), 'year')
        : isAnotherStartYear;
    const endDateFormat = getFormatWithNbsp(
        getFormatForInterval(isAnotherEndYear, short),
    );

    if (startMoment.isSame(endMoment, 'month')) {
        return `${startMoment.date()}${CHAR_NBSP}${CHAR_MIDDLE_DASH} ${endMoment.format(
            endDateFormat,
        )}`;
    }

    return `${startMoment.format(
        stateDateFormat,
    )}${CHAR_NBSP}${CHAR_MIDDLE_DASH} ${endMoment.format(endDateFormat)}`;
}

export function humanizeDate(date: DateLikeType): string {
    const dateMoment = moment(date);
    const isAnotherYear = dateMoment.isAfter(moment(), 'year');
    const format = getFormatWithNbsp(getFormatForDate(isAnotherYear));

    return dateMoment.format(format);
}

function getFormatForDate(isAnotherYear: boolean): string {
    return isAnotherYear ? HUMAN_WITH_YEAR : HUMAN;
}

function getFormatForInterval(isAnotherYear: boolean, short: boolean): string {
    if (isAnotherYear) {
        return short ? HUMAN_SHORT_WITH_SHORT_YEAR : HUMAN_WITH_YEAR;
    }

    return short ? HUMAN_SHORT : HUMAN;
}

function getFormatWithNbsp(format: string): string {
    return format.replace(/\s+/g, CHAR_NBSP);
}
