import moment, {Moment} from 'moment';

import {ETimeOfDay} from 'utilities/dateUtils/types';

import locales from 'utilities/dateUtils/locales';
import localesOverrides from 'utilities/dateUtils/localesOverrides';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import {
    HUMAN_SHORT_WITH_YEAR,
    HUMAN_SHORT,
    HUMAN,
    HUMAN_WITH_YEAR,
    HUMAN_DATE_RU,
} from './formats';

/**
 * number - Количество миллисекунд, прошедших с 1 января 1970 года.
 * string - Любая строка, которая преобразуется в дату.
 */
// eslint-disable-next-line @typescript-eslint/naming-convention
export type DateLikeType = number | string | Date | Moment;

/**
 * Перезаписывает настройки локали, если требуется
 *
 * @param {string} locale - язык
 *
 * @return {void}
 */
export function updateLocaleIfNeeded(locale: string): void {
    if (locale === locales.RU) {
        moment.updateLocale(locale, localesOverrides[locales.RU]);
    }
}

/**
 * Устанавливает язык для дат.
 *
 * @param {string} locale - Язык.
 *
 * @return {string}
 */
export function setDateLocale(locale: string): string {
    updateLocaleIfNeeded(locale);

    return moment.locale(locale);
}

/**
 * Возвращает текущий язык.
 *
 * @return {string}
 */
export function getDateLocale(): string {
    return moment.locale();
}

/**
 * Возвращает количество миллисекунд прошедших с 1 января 1970 года до сегодняшнего дня.
 * @return {number}
 */
export function getNow(): number {
    let now = Date.now();

    if (typeof window !== 'undefined' && window.timeCorrection) {
        now += window.timeCorrection;
    }

    return now;
}

// Фугнкция вернет момент начала дня
export function getStartOfDay(date: DateLikeType): Moment {
    return moment(date).startOf('day');
}

/**
 * Преобразование строки или нативного объекта Date в формат даты, с которым работает приложение.
 *
 * @param {DateLikeType} date - Дата, которую нужно обработать.
 * @param {string} [format] - Формат даты, если нужно обработать строку.
 *
 * @return {Moment}
 */
export function parseDate(date: DateLikeType, format?: string): Moment {
    return moment(date, format);
}

/**
 * Является ли дата валидной.
 *
 * @param date - Дата, которую нужно проверить.
 * @param format - Формат даты
 *
 * @return {boolean}
 */
export function isValidDate(date: DateLikeType, format?: string): boolean {
    return moment(date, format).isValid();
}

/**
 * Отображение даты в необходимом формате.
 *
 * @param date - Дата, которую нужно преобразовать в строку.
 * @param format - Формат даты.
 * @param options - Настройки форматирования.
 */
export function formatDate(
    date: DateLikeType,
    format: string,
    options?: {
        withNbsp?: boolean;
    },
): string {
    const withNbsp = options?.withNbsp;

    const formattedDate = moment(date).format(format);

    return withNbsp ? formattedDate.replace(/\s/g, CHAR_NBSP) : formattedDate;
}

/**
 * Отображение даты в человекочитаемом виде.
 * Если год даты совпадает с текущим годом, то год не отображается.
 *
 * @param {DateLikeType} date - Дата, которую нужно преобразовать в строку.
 *
 * @return {string}
 */
export function humanFormatDate(date: DateLikeType): string {
    const isCurrentYear = moment(date).isSame(moment(), 'year');
    const yearIsShown = !isCurrentYear;

    return formatDate(date, yearIsShown ? HUMAN_WITH_YEAR : HUMAN);
}

export interface IFormatDateRangeParams {
    delimiter?: string;
    forceHideFromDateYearInRange?: boolean;
}

/**
 * Отображение диапазона времени в человекочитаемом виде.
 * Если дата окончания не задана, то будет выведена только дата начала.
 *
 * @param {DateLikeType} from - Дата начала.
 * @param {DateLikeType} [to] - Дата окончания.
 * @param {IFormatDateRangeParams} [params] - Доп параметры.
 *
 * @return {string}
 */
export function formatDateRange(
    from: DateLikeType,
    to?: DateLikeType,
    params?: IFormatDateRangeParams,
): string {
    if (!to) {
        return humanFormatDate(from);
    }

    const {delimiter = ' — ', forceHideFromDateYearInRange} = params || {};
    const fromM = moment(from);
    const toM = moment(to);
    const isSameYear = fromM.isSame(toM, 'year');
    const isCurrentYear = fromM.isSame(moment(), 'year');
    const yearIsShown = !isSameYear || !isCurrentYear;

    const hideFromDate = Boolean(forceHideFromDateYearInRange && to);

    const formattedFrom = formatDate(
        from,
        yearIsShown && !hideFromDate ? HUMAN_SHORT_WITH_YEAR : HUMAN_SHORT,
    );
    const formattedTo = formatDate(
        to,
        yearIsShown ? HUMAN_SHORT_WITH_YEAR : HUMAN_SHORT,
    );

    return `${formattedFrom}${delimiter}${formattedTo}`;
}

/**
 * Проверяем дату, чтобы бекенд корректно мог ее вставить в таблицу
 * Следующие требования: дата валидна, год 1900 и выше и год меньше 10000
 *
 * @param {DateLikeType} date - Дата
 *
 * @return {boolean}
 */
export function isValidDateForBackend(date: DateLikeType) {
    const momentDate = moment(date, HUMAN_DATE_RU);

    if (!isValidDate(momentDate)) {
        return false;
    }

    const dateMin = moment('1900-01-01');
    const dateMax = moment('9999-12-31');

    return !momentDate.isBefore(dateMin) && !momentDate.isAfter(dateMax);
}

/**
 * Проверить, в будущем ли дата
 * Вернет true, если в будущем и false, если нет
 *
 * @param {DateLikeType} date - дата, которую проверяем
 *
 * @return {boolean}
 */
export function isDateInFuture(date: DateLikeType) {
    return moment(date, HUMAN_DATE_RU).isAfter(moment());
}

export function getTimeOfDay(hour: string | number): ETimeOfDay {
    const hourN = Number(hour);

    switch (true) {
        case hourN >= 0 && hourN < 6:
            return ETimeOfDay.NIGHT;

        case hourN >= 6 && hourN < 12:
            return ETimeOfDay.MORNING;

        case hourN >= 12 && hourN < 18:
            return ETimeOfDay.DAY;

        case hourN >= 18 && hourN <= 23:
        default:
            return ETimeOfDay.EVENING;
    }
}

export function diffDays(dayA: DateLikeType, dayB: DateLikeType) {
    return parseDate(dayA)
        .startOf('day')
        .diff(parseDate(dayB).startOf('day'), 'days');
}
