type TTimeoutType = ReturnType<typeof setTimeout> | undefined;

/**
 * Возвращает функцию, с debounce для каждой сущности.
 * Сущность определяется первым параметром в передаваемой в func аргументом
 *
 * @param func
 * @param wait
 *
 * @example
 * function testFunc(id, value) {
 *     console.log(`id = ${id}, value = ${value}`);
 * }
 * const debounced = debounce(testFunc, 2000);
 *
 * debounced('id1', 200);
 * debounced('id1', 300);
 * debounced('id1', 400); // <- сработает этот
 * debounced('id2', 20);
 * debounced('id2', 30); // <- сработает этот
 */
export default function debounceById<Args extends unknown[]>(
    func: (...args: Args) => void,
    wait: number,
): (id: string, ...args: Args) => void {
    const reviewTimeouts: {
        [entityId: string]: TTimeoutType;
    } = {};

    return function (id, ...args): void {
        let timeout = reviewTimeouts[id];

        /* Вызов функции через 'wait' ms */
        const later = (): void => {
            timeout = undefined;
            func(...args);
        };

        if (timeout) {
            clearTimeout(timeout);
        }

        timeout = setTimeout(later, wait);

        reviewTimeouts[id] = timeout;
    };
}
